/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */
import { useState } from "preact/hooks";

export interface Options {
  slowTolerance: number;
}

export interface AsyncOperationApi<T> {
  request: (...a: unknown[]) => void;
  cancel: () => void;
  data: T | undefined;
  isSlow: boolean;
  isLoading: boolean;
  error: string | undefined;
}

export function useAsync<T>(
  fn?: (...args: unknown[]) => Promise<T>,
  { slowTolerance: tooLong }: Options = { slowTolerance: 1000 },
): AsyncOperationApi<T> {
  const [data, setData] = useState<T | undefined>(undefined);
  const [isLoading, setLoading] = useState<boolean>(false);
  const [error, setError] = useState<string>();
  const [isSlow, setSlow] = useState(false);

  const request = async (...args: unknown[]) => {
    if (!fn) return;
    setLoading(true);

    const handler = setTimeout(() => {
      setSlow(true);
    }, tooLong);

    try {
      const result = await fn(...args);
      setData(result);
    } catch (error) {
      setError(error instanceof Error ? error.message : String(error));
    }
    setLoading(false);
    setSlow(false);
    clearTimeout(handler);
  };

  function cancel(): void {
    setLoading(false);
    setSlow(false);
  }

  return {
    request,
    cancel,
    data,
    isSlow,
    isLoading,
    error,
  };
}
