// @ts-ignore: no type decl for this library
import * as jedLib from "jed";

export let jed: any = undefined;

/**
 * Set up jed library for internationalization,
 * based on browser language settings.
 */
export function setupI18n(lang: string, strings: { [s: string]: any }): void {
  lang = lang.replace("_", "-");

  if (!strings[lang]) {
    strings[lang] = {};
  }
  jed = new jedLib.Jed(strings[lang]);
}

/**
 * Use different translations for testing.  Should not be used outside
 * of test cases.
 */
export function internalSetStrings(langStrings: any): void {
  jed = new jedLib.Jed(langStrings);
}

declare const __translated: unique symbol;
export type TranslatedString = string & { [__translated]: true };
export type ToTranslateString = string & { [__translated]: true };

/**
 * Convert template strings to a msgid
 */
function toI18nString(stringSeq: ReadonlyArray<string>): TranslatedString {
  let s = "";
  for (let i = 0; i < stringSeq.length; i++) {
    s += stringSeq[i];
    if (i < stringSeq.length - 1) {
      s += `%${i + 1}$s`;
    }
  }
  return s as TranslatedString;
}

/**
 * Internationalize a string template with arbitrary serialized values.
 */
export function singular(
  stringSeq: TemplateStringsArray,
  ...values: any[]
): TranslatedString {
  const s = toI18nString(stringSeq);
  // jed throws a Error when key is empty
  if (!s) return "" as TranslatedString;
  const tr = jed
    .translate(s)
    .ifPlural(1, s)
    .fetch(...values);
  return tr;
}

function withContext(ctx: string): typeof singular {
  return function (t: TemplateStringsArray, ...v: any[]): TranslatedString {
    const s = toI18nString(t);
    const tr = jed
      .translate(s)
      .withContext(ctx)
      .ifPlural(1, s)
      .fetch(...v);
    return tr;
  };
}

/**
 * Internationalize a string template without serializing
 */
export function translate(
  stringSeq: TemplateStringsArray,
  ...values: any[]
): TranslatedString[] {
  const s = toI18nString(stringSeq);
  if (!s) return [];
  const translation: TranslatedString = jed.ngettext(s, s, 1);
  return replacePlaceholderWithValues(translation, values);
}

/**
 * Internationalize a string template without serializing
 */
export function Translate({
  children,
  debug,
  context: ctx,
}: {
  children: any;
  debug?: boolean;
  context?: string;
}): any {
  const c = [].concat(children);
  const s = stringifyArray(c);
  if (!s) return [];
  const translation: TranslatedString = ctx
    ? jed.npgettext(ctx, s, s, 1)
    : jed.ngettext(s, s, 1);
  if (debug) {
    console.log("looking for ", s, "got", translation);
  }
  return replacePlaceholderWithValues(translation, c);
}

/**
 * Get an internationalized string (based on the globally set, current language)
 * from a JSON object.  Fall back to the default language of the JSON object
 * if no match exists.
 */
export function getJsonI18n<K extends string>(
  obj: Record<K, string>,
  key: K,
): string {
  return obj[key];
}

export function getTranslatedArray(array: Array<any>) {
  const s = stringifyArray(array);
  const translation: TranslatedString = jed.ngettext(s, s, 1);
  return replacePlaceholderWithValues(translation, array);
}

function replacePlaceholderWithValues(
  translation: TranslatedString,
  childArray: Array<any>,
): Array<any> {
  const tr = translation.split(/%(\d+)\$s/);
  // const childArray = toChildArray(children);
  // Merge consecutive string children.
  const placeholderChildren = [];
  for (let i = 0; i < childArray.length; i++) {
    const x = childArray[i];
    if (x === undefined) {
      continue;
    } else if (typeof x === "string") {
      continue;
    } else {
      placeholderChildren.push(x);
    }
  }
  const result = [];
  for (let i = 0; i < tr.length; i++) {
    if (i % 2 == 0) {
      // Text
      result.push(tr[i]);
    } else {
      const childIdx = Number.parseInt(tr[i]) - 1;
      result.push(placeholderChildren[childIdx]);
    }
  }
  return result;
}

function stringifyArray(children: Array<any>): string {
  let n = 1;
  const ss = children.map((c) => {
    if (typeof c === "string") {
      return c;
    }
    return `%${n++}$s`;
  });
  const s = ss.join("").replace(/ +/g, " ").trim();
  return s;
}

export type InternationalizationAPI = typeof i18n;
export type Translator = (i18n: InternationalizationAPI) => TranslatedString;

export const i18n = {
  str: singular,
  ctx: withContext,
  singular,
  Translate,
  translate,
};
