/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_select_order_blinded_sigs.c
 * @brief Implementation of the select blinded sigs by order_id function for Postgres
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_select_order_blinded_sigs.h"
#include "pg_helper.h"

/**
 * @brief Context for the callback to handle the result of the blinded sigs selection.
 */
struct SelectBlindedSigsContext
{
  /**
   * @brief Callback to be called with the result of the blinded sigs selection.
   */
  void *cb_cls;

  /**
   * @brief Callback to be called with the result of the blinded sigs selection.
   */
  TALER_MERCHANTDB_BlindedSigCallback cb;

  /**
   * @brief Result status of the query.
   */
  enum GNUNET_DB_QueryStatus qs;
};


/**
 * @brief Callback to handle the result of the blinded sigs selection.
 *
 * @param cls the closure containing the callback and its context
 * @param result the result of the query
 * @param num_results number of results in the result set
 */
static void
restore_sig_cb (void *cls,
                PGresult *result,
                unsigned int num_results)
{
  struct SelectBlindedSigsContext *ctx = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    struct GNUNET_HashCode h_issue;
    struct GNUNET_CRYPTO_BlindedSignature *sig;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_auto_from_type ("token_hash",
                                            &h_issue),
      GNUNET_PQ_result_spec_blinded_sig ("token_blinded_signature",
                                         &sig),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      ctx->qs = GNUNET_DB_STATUS_HARD_ERROR;
      return;
    }
    ctx->cb (ctx->cb_cls,
             &h_issue,
             sig);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_select_order_blinded_sigs (
  void *cls,
  const char *order_id,
  TALER_MERCHANTDB_BlindedSigCallback cb,
  void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct SelectBlindedSigsContext ctx = {
    .cb = cb,
    .cb_cls = cb_cls,
    .qs = GNUNET_DB_STATUS_SUCCESS_NO_RESULTS
  };

  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (order_id),
    GNUNET_PQ_query_param_end
  };

  check_connection (pg);
  PREPARE (pg,
           "select_blinded_sigs",
           "SELECT"
           "  motbs.token_blinded_signature"
           " ,motbs.token_hash"
           " FROM merchant_order_token_blinded_sigs AS motbs"
           " JOIN merchant_contract_terms AS mct USING (order_serial)"
           " WHERE mct.order_id = $1"
           " ORDER BY motbs.token_index ASC");
  return GNUNET_PQ_eval_prepared_multi_select (
    pg->conn,
    "select_blinded_sigs",
    params,
    &restore_sig_cb,
    &ctx);
}
