# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000-2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

class Intervall:
    """
    The intervall is internally a intervall less than octave
    pluss n octaves. The data variables:
      m_dir
      m_octave
      m_intervall
      m_mod
    should NOT be touched by anyone, except MusicalPitch.__add__
    """
    def __init__(self):
        self.m_dir = 1 # value as to be 1 or -1 for initialised obj
        self.m_octave = 0
        self.m_intervall = 0 # 0:prim, 1:seond, ... 6: septim
        # second:  -2:dim -1:minor 0:major   1:aug
        # third:      dim    minor   major     aug
        # fourth:              dim   perfect   aug
        # fifth:               dim   perfect   aug
        # sixth:      dim    minor   major     aug
        # seventh:    dim    minor   major     aug
        self.m_mod = 0
    def errorcheck(self):
        assert 0 <= self.m_intervall <= 6
        assert -2 <= self.m_mod <= 1 # should be increased to -3 <= x <= 2
        assert self.m_dir in (-1, 1)
    def _set(self, dir, intervall, mod, octave):
        self.m_dir = dir
        self.m_intervall = intervall
        self.m_mod = mod
        self.m_octave = octave
        self.errorcheck()
        return self
    def set_from_int(self, i):
        """It returns self to allow chaining: set_from_int(4).pretty_name()
        """
        if i < 0:
            self.m_dir = -1
        else:
            self.m_dir = 1
        self.m_octave = abs(i) / 12
        self.m_mod, self.m_intervall = ((0, 0), #prim
               (-1, 1), (0, 1), # second
               (-1, 2), (0, 2), # third
               (0, 3),          # fourth
               (-1, 4),         # dim 5, tritonus
               (0, 4),          # fifth
               (-1, 5), (0, 5), # sixth
               (-1, 6), (0, 6))[abs(i) % 12] # seventh
        return self
    def get_intvalue(self):
        self.errorcheck()
        return ([0, 2, 4, 5, 7, 9, 11][self.m_intervall] + self.m_octave * 12 + self.m_mod) * self.m_dir
    def __str__(self):
        self.errorcheck()
        s = "(Intervall %i %imod %io" % (self.m_intervall, self.m_mod, self.m_octave)
        if self.m_dir == -1:
            s = s + " down)"
        else:
            s = s + " up)"
        return s

if __name__=="__main__":
    import musicalpitch
    print "testing the intervall module"
    i = Intervall()
    for int, intervall, modifier, octave, dir in (
          (5, 3, 0, 0, 1), (1, 1, -1, 0, 1), (2, 1, 0, 0, 1),
          (-1, 1, -1, 0, -1), (-2, 1, 0, 0, -1)):
        i.set_from_int(int)
        print "set_from_int(%i)" % int
        if i.m_intervall == intervall:
            print "intervall:%i, " % intervall,
        else:
            print "intervall: failed:%i!=%i, " % (i.m_intervall, intervall),
        if i.m_mod == modifier:
            print "mod:%i, " % modifier,
        else:
            print "mod: failed:%i=!%i, " % (i.m_mod, modifier),
        if i.m_octave == octave:
            print "octave:%i, " % octave,
        else:
            print "octave: failed:%i=!%i, " % (i.m_octave, octave),
        if i.m_dir == dir:
            print "dir:%i, " % dir,
        else:
            print "dir: failed:%i=!%i, " % (i.m_dir, dir),
        print
    for old, new, i in (("c'", "d'", 2), ("e'", "fis'", 2), ("e", "b,", -5),
           ("e", "bes,", -6), ("e", "ais,", -6)):
        n = musicalpitch.MusicalPitch(old) + Intervall().set_from_int(i)
        if n.str() == new:
            print "ok: %s + %i == %s" % (old, i, new)
        else:
            print "%s + %i == %s failed:" % (old, i, new), n.str()
