/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6_H
#define LIQUIDWAR6_H

/*
 * This is the main Liquid War 6 header file.
 */

#define LW6_VERSION "0.2.3551"

#include <pthread.h>
#include <libguile.h>

#include "sys/sys.h"
#include "glb/glb.h"
#include "dyn/dyn.h"
#include "cns/cns.h"
#include "hlp/hlp.h"
#include "cfg/cfg.h"
#include "map/map.h"
#include "ker/ker.h"
#include "pil/pil.h"
#include "bot/bot.h"
#include "sim/sim.h"
#include "gen/gen.h"
#include "ldr/ldr.h"
#include "tsk/tsk.h"
#include "mat/mat.h"
#include "gui/gui.h"
#include "gfx/gfx.h"
#include "dsp/dsp.h"
#include "snd/snd.h"
#include "img/img.h"
#include "net/net.h"
#include "nod/nod.h"
#include "cnx/cnx.h"
#include "msg/msg.h"
#include "cli/cli.h"
#include "srv/srv.h"
#include "dat/dat.h"
#include "p2p/p2p.h"
#include "scm/scm.h"

/**
 * List of all available smob types.
 */
typedef struct lw6_smob_types_s
{
  /// Id/key for dsp smob type.
  scm_t_bits dsp;
  /// Id/key for snd smob type.
  scm_t_bits snd;
  /// Id/key for map smob type.
  scm_t_bits map;
  /// Id/key for menu smob type.
  scm_t_bits menu;
  /// Id/key for game_struct smob type.
  scm_t_bits game_struct;
  /// Id/key for game_state smob type.
  scm_t_bits game_state;
  /// Id/key for pilot smob type.
  scm_t_bits pilot;
  /// Id/key for bot smob type.
  scm_t_bits bot;
  /// Id/key for look smob type.
  scm_t_bits look;
  /// Id/key for loader smob type.
  scm_t_bits loader;
  /// Id/key for db smob type.
  scm_t_bits db;
  /// Id/key for node smob type.
  scm_t_bits node;
  /// Id/key for jpeg smob type.
  scm_t_bits jpeg;
}
lw6_smob_types_t;

/**
 * Data for dsp smob.
 */
typedef struct lw6_dsp_smob_s
{
  /// C object this smob refers to.
  lw6dsp_backend_t *c_dsp;
  /// Guile level object this smob depends on.
  SCM level;
  /// Guile game_struct object this smob depends on.
  SCM game_struct;
  /// Guile game_state object this smob depends on.
  SCM game_state;
  /// Guile pilot object this smob depends on.
  SCM pilot;
}
lw6_dsp_smob_t;

/**
 * Data for snd smob.
 */
typedef struct lw6_snd_smob_s
{
  /// C object this smob refers to.
  lw6snd_backend_t *c_snd;
}
lw6_snd_smob_t;

/**
 * Data for map smob.
 */
typedef struct lw6_map_smob_s
{
  /// C object this smob refers to.
  lw6map_level_t *c_map;
}
lw6_map_smob_t;

/**
 * Data for menu smob.
 */
typedef struct lw6_menu_smob_s
{
  /// C object this smob refers to.
  lw6gui_menu_t *c_menu;
}
lw6_menu_smob_t;

/**
 * Data for game_struct smob.
 */
typedef struct lw6_game_struct_smob_s
{
  /// C object this smob refers to.
  lw6ker_game_struct_t *c_game_struct;
  /// Guile map object this smob depends on.
  SCM map;
}
lw6_game_struct_smob_t;

/**
 * Data for game_state smob.
 */
typedef struct lw6_game_state_smob_s
{
  /// C object this smob refers to.
  lw6ker_game_state_t *c_game_state;
  /// Guile game_struct object this smob depends on.
  SCM game_struct;
}
lw6_game_state_smob_t;

/**
 * Data for pilot smob.
 */
typedef struct lw6_pilot_smob_s
{
  /// C object this smob refers to.
  lw6pil_pilot_t *c_pilot;
}
lw6_pilot_smob_t;

/**
 * Data for bot smob.
 */
typedef struct lw6_bot_smob_s
{
  /// C object this smob refers to.
  lw6bot_backend_t *c_bot;
  /// Guile game_state object this smob depends on.
  SCM game_state;
  /// Guile pilot object this smob depends on.
  SCM pilot;
}
lw6_bot_smob_t;

/**
 * Data for look smob.
 */
typedef struct lw6_look_smob_s
{
  /// C object this smob refers to.
  lw6gui_look_t *c_look;
}
lw6_look_smob_t;

/**
 * Data for loader smob.
 */
typedef struct lw6_loader_smob_s
{
  /// C object this smob refers to.
  lw6tsk_loader_t *c_loader;
}
lw6_loader_smob_t;

/**
 * Data for db smob.
 */
typedef struct lw6_db_smob_s
{
  /// C object this smob refers to.
  lw6p2p_db_t *c_db;
}
lw6_db_smob_t;

/**
 * Data for node smob.
 */
typedef struct lw6_node_smob_s
{
  /// C object this smob refers to.
  lw6p2p_node_t *c_node;
  /// Guile db object this smob depends on.
  SCM db;
}
lw6_node_smob_t;

/**
 * Data for jpeg smob.
 */
typedef struct lw6_jpeg_smob_s
{
  /// C object this smob refers to.
  lw6img_jpeg_t *c_jpeg;
}
lw6_jpeg_smob_t;

/**
 * Global game data. This structure is used to store
 * persistent game state.
 */
typedef struct lw6_global_s
{
  /**
   * Mutex used to protect some objects when
   * concurrent accesses are done. This is very likely
   * to happen as the game is multithreaded.
   */
  pthread_mutex_t mutex;
  /// Configuration information (config file access).
  void *cfg_context;
  /// Wether libcfg has been initialized.
  int cfg_initialized;
  /// Wether libcns has been initialized.
  int cns_initialized;
  /// Wether libnet has been initialized.
  int net_initialized;
  /// Game state time (in seconds).
  int start_time;
  /// Argc as passed to main.
  int argc;
  /// Argv as passed to main.
  const char **argv;
  /// Progress indicator used by GUI.
  float progress;
  /// Collection of smob types definitions.
  lw6_smob_types_t smob_types;
  /// Assoc with all dsp smobs.
  lw6sys_assoc_t *dsp_smobs;
  /// Assoc with all snd smobs.
  lw6sys_assoc_t *snd_smobs;
  /// Assoc with all map smobs.
  lw6sys_assoc_t *map_smobs;
  /// Assoc with all menu smobs.
  lw6sys_assoc_t *menu_smobs;
  /// Assoc with all game_struct smobs.
  lw6sys_assoc_t *game_struct_smobs;
  /// Assoc with all game_state smobs.
  lw6sys_assoc_t *game_state_smobs;
  /// Assoc with all pilot smobs.
  lw6sys_assoc_t *pilot_smobs;
  /// Assoc with all bot smobs.
  lw6sys_assoc_t *bot_smobs;
  /// Assoc with all look smobs.
  lw6sys_assoc_t *look_smobs;
  /// Assoc with all loader smobs.
  lw6sys_assoc_t *loader_smobs;
  /// Assoc with all db smobs.
  lw6sys_assoc_t *db_smobs;
  /// Assoc with all node smobs.
  lw6sys_assoc_t *node_smobs;
  /// Assoc with all jpeg smobs.
  lw6sys_assoc_t *jpeg_smobs;
  /**
   * Coverage hash, keeps a track of all calls to script functions,
   * this way one can figure out wether a C function exported to
   * script has been called, and how many times.
   */
  lw6sys_hash_t *coverage;
  /// Return value of the called script.
  int ret;
}
lw6_global_t;

/* lw6-callback.c */
extern void lw6_resize_callback (lw6gui_video_mode_t * video_mode);

/* lw6-control.c */
extern void lw6_release ();
extern void lw6_exit ();
extern void lw6_set_ret (int ret);
extern int lw6_get_ret ();

/* lw6-fix.c */
extern int lw6_fix_env (int argc, const char *argv[]);

/* lw6-funcs.c */
extern int lw6_register_funcs ();
extern void lw6_cns_handler (char *line);

/* lw6-funcssys.c */
extern int lw6_register_funcs_sys ();

/* lw6-funcshlp.c */
extern int lw6_register_funcs_hlp ();

/* lw6-funcscfg.c */
extern int lw6_register_funcs_cfg ();

/* lw6-funcsgui.c */
extern int lw6_register_funcs_gui ();

/* lw6-funcsgfx.c */
extern int lw6_register_funcs_gfx ();

/* lw6-funcsdsp.c */
extern int lw6_register_funcs_dsp ();

/* lw6-funcsmap.c */
extern int lw6_register_funcs_map ();

/* lw6-funcsker.c */
extern int lw6_register_funcs_ker ();

/* lw6-funcsgen.c */
extern int lw6_register_funcs_gen ();

/* lw6-funcsldr.c */
extern int lw6_register_funcs_ldr ();

/* lw6-funcspil.c */
extern int lw6_register_funcs_pil ();

/* lw6-funcssnd.c */
extern int lw6_register_funcs_snd ();

/* lw6-funcscns.c */
extern int lw6_register_funcs_cns ();

/* lw6-funcstsk.c */
extern int lw6_register_funcs_tsk ();

/* lw6-funcsimg.c */
extern int lw6_register_funcs_img ();

/* lw6-funcsnet.c */
extern int lw6_register_funcs_net ();

/* lw6-funcscli.c */
extern int lw6_register_funcs_cli ();

/* lw6-funcssrv.c */
extern int lw6_register_funcs_srv ();

/* lw6-funcsp2p.c */
extern int lw6_register_funcs_p2p ();

/* lw6-funcsbot.c */
extern int lw6_register_funcs_bot ();

/* lw6-global.c */
#define LW6_MUTEX_LOCK pthread_mutex_lock( &lw6_global.mutex );
#define LW6_MUTEX_UNLOCK pthread_mutex_unlock( &lw6_global.mutex );

extern lw6_global_t lw6_global;
extern int lw6_init_global (int argc, const char *argv[]);
extern void lw6_quit_global ();

/* lw6-main.c */
extern int lw6_main (int argc, const char *argv[]);

/* lw6-options.c */
extern int lw6_process_non_run_options (int argc, const char *argv[],
					int *run_game);

/* lw6-smobs.c */
extern SCM lw6_make_scm_dsp (lw6dsp_backend_t * c_dsp);
extern lw6dsp_backend_t *lw6_scm_to_dsp (SCM dsp);
extern void lw6_free_dsp_smob (lw6_dsp_smob_t * dsp_smob);
extern SCM lw6_make_scm_snd (lw6snd_backend_t * c_snd);
extern lw6snd_backend_t *lw6_scm_to_snd (SCM snd);
extern void lw6_free_snd_smob (lw6_snd_smob_t * snd_smob);
extern SCM lw6_make_scm_map (lw6map_level_t * c_map);
extern lw6map_level_t *lw6_scm_to_map (SCM map);
extern void lw6_free_map_smob (lw6_map_smob_t * map_smob);
extern SCM lw6_make_scm_menu (lw6gui_menu_t * c_menu);
extern lw6gui_menu_t *lw6_scm_to_menu (SCM menu);
extern void lw6_free_menu_smob (lw6_menu_smob_t * menu_smob);
extern SCM lw6_make_scm_game_struct (lw6ker_game_struct_t * c_game_struct,
				     SCM map);
extern lw6ker_game_struct_t *lw6_scm_to_game_struct (SCM game_struct);
extern void lw6_free_game_struct_smob (lw6_game_struct_smob_t *
				       game_struct_smob);
extern SCM lw6_make_scm_game_state (lw6ker_game_state_t * c_game_state,
				    SCM game_struct);
extern lw6ker_game_state_t *lw6_scm_to_game_state (SCM game_state);
extern void lw6_free_game_state_smob (lw6_game_state_smob_t *
				      game_state_smob);
extern SCM lw6_make_scm_pilot (lw6pil_pilot_t * c_pilot);
extern lw6pil_pilot_t *lw6_scm_to_pilot (SCM pilot);
extern void lw6_free_pilot_smob (lw6_pilot_smob_t * pilot_smob);
extern SCM lw6_make_scm_bot (lw6bot_backend_t * c_bot, SCM game_state,
			     SCM pilot);
extern lw6bot_backend_t *lw6_scm_to_bot (SCM bot);
extern void lw6_free_bot_smob (lw6_bot_smob_t * bot_smob);
extern SCM lw6_make_scm_look (lw6gui_look_t * c_look);
extern lw6gui_look_t *lw6_scm_to_look (SCM look);
extern void lw6_free_look_smob (lw6_look_smob_t * look_smob);
extern SCM lw6_make_scm_loader (lw6tsk_loader_t * c_loader);
extern lw6tsk_loader_t *lw6_scm_to_loader (SCM loader);
extern void lw6_free_loader_smob (lw6_loader_smob_t * loader_smob);
extern SCM lw6_make_scm_db (lw6p2p_db_t * c_db);
extern lw6p2p_db_t *lw6_scm_to_db (SCM db);
extern void lw6_free_db_smob (lw6_db_smob_t * db_smob);
extern SCM lw6_make_scm_node (lw6p2p_node_t * c_node, SCM db);
extern lw6p2p_node_t *lw6_scm_to_node (SCM node);
extern void lw6_free_node_smob (lw6_node_smob_t * node_smob);
extern SCM lw6_make_scm_jpeg (lw6img_jpeg_t * c_jpeg);
extern lw6img_jpeg_t *lw6_scm_to_jpeg (SCM jpeg);
extern void lw6_free_jpeg_smob (lw6_jpeg_smob_t * jpeg_smob);
extern int lw6_register_smobs ();

/* lw6-test.c */
extern int lw6_test_register (int mode);
extern int lw6_test_run (int mode);

#endif
