package kawa.lang;
import kawa.standard.Scheme;
import gnu.bytecode.Method;
import gnu.bytecode.Variable;
import gnu.mapping.*;
import gnu.expr.*;
import java.lang.reflect.Modifier;
import gnu.bytecode.Type;
import gnu.bytecode.ClassType;
import gnu.text.SourceMessages;

/** Used to translate from source to Expression.
 * The result has macros expanded, lexical names bound, etc, and is
 * ready for code generation.
 * This is sometimes called a "compilation environment",
 * but we modify it as we go along - there is a single Translator for
 * each top-level form.
 */

public class Translator extends Object
{
  // Map name to Declaration.
  public Environment current_decls;
  ScopeExp current_scope;

  // This is null in JDK 1.1 and something else in JDK 1.2.
  private ClassLoader systemClassLoader = getClass().getClassLoader();

  public LambdaExp currentLambda () { return current_scope.currentLambda (); }

  public ScopeExp currentScope() { return current_scope; }

  /** Return true if decl is lexical and not fluid. */
  public boolean isLexical (Declaration decl)
  {
    if (decl == null)
      return false;
    if (! decl.isFluid())
      return true;
    ScopeExp scope = currentScope();
    ScopeExp context = decl.getContext();
    for (;; scope = scope.outer)
      {
	if (scope == null)
	  return false;
	if (scope == context)
	  return true;
	if (scope instanceof LambdaExp
	    && ! ((LambdaExp) scope).getInlineOnly())
	  return false;
      }
  }

  private static Expression errorExp = new ErrorExp ("unknown syntax error");
  String current_filename;
  int current_line;
  int current_column;

  // Global environment used to look for syntax/macros.
  private Environment env;

  public Translator (Environment env, SourceMessages messages)
  {
    this.env = env;
    current_decls = new Environment();
    this.messages = messages;
  }

  public Translator (Environment env)
  {
    this.env = env;
    current_decls = new Environment();
    messages = new SourceMessages();
  }

  public Translator ()
  {
    this (Environment.user());
  }

  public final Environment getGlobalEnvironment() { return env; }

  final Expression rewrite_car (Pair pair)
  {
    Object car = pair.car;
    if (pair instanceof PairWithPosition)
      return rewrite_with_position (car, (PairWithPosition) pair);
    else
      return rewrite (car);
  }

  Syntax currentSyntax;
  public Syntax getCurrentSyntax() { return currentSyntax; }

  /**
   * Apply a Syntax object.
   * @param syntax the Syntax object whose rewrite method we call
   * @param form the syntax form (including the macro name)
   * @return the re-written form as an Expression object
   */
  Expression apply_rewrite (Syntax syntax, Pair form)
  {
    Expression exp = errorExp;
    Syntax saveSyntax = currentSyntax;
    currentSyntax = syntax;
    try
      {
	exp = syntax.rewriteForm(form, this);
      }
    finally
      {
        currentSyntax = currentSyntax;
      }
    return exp;
  }

  SourceMessages messages;

  public SourceMessages getMessages() { return messages; }
  public void setMessages (SourceMessages messages)
  { this.messages = messages; }
 
  public void error(char severity, String message)
  {
    messages.error(severity, current_filename, current_line, current_column,
		   message);
  }


  /**
   * Handle syntax errors (at rewrite time).
   * @param message an error message to print out
   * @return an ErrorExp
   */
  public Expression syntaxError (String message)
  {
    error('e', message);
    return new ErrorExp (message);
  }

  /** Enter a global definition.
   * This allows macro definitions to be used in the same Translation
   * as the define-syntax.
   */

  public void addGlobal (String name, Object value)
  {
    env.put (name, value);
  }

  /** Note that we have seen a construct that must be compiled, not evaluated.
   * If we are not inside a lambda (which is always compiled), but
   * only inside the outer-most ModuleExp, note that it must be compiled. */
  public void mustCompileHere ()
  {
    LambdaExp lambda = currentLambda ();
    if (lambda instanceof ModuleExp)
      ((ModuleExp)lambda).mustCompile = true;
  }

  Object getBinding (Object obj)
  {
    if (obj instanceof String)
      {
	String sym = (String) obj;
	obj = current_decls.get (sym);

        if (obj instanceof Syntax)
          return obj;
	Binding binding = null;
        if (obj != null)
	  {
	    // Hygenic macro expansion may bind a renamed (uninterned) symbol
	    // to the original symbol.  Here, use the original symbol.
	    if (obj instanceof String)
	      binding = env.lookup((String) obj);
	    else if (obj instanceof Declaration
		     && ! isLexical((Declaration) obj))
	      obj = null;
	  }
	else
	  binding = env.lookup(sym);
	if (binding != null && binding.isBound())
	  return binding.get();
	return null;
      }
     return obj;
  }

  /** Check if Object is Syntax, or bound to Syntax.
   * @param obj the value to check
   * @return the Syntax bound to obj, or null.
   */
  public Syntax check_if_Syntax (Object obj)
  {
    obj = getBinding(obj);
    if (obj instanceof Syntax)
      return (Syntax) obj;
    return null;
  }

  public Expression rewrite_pair (Pair p)
  {
    if (p.car instanceof Syntax)
      return apply_rewrite((Syntax) p.car, p);
    Object cdr = p.cdr;

    Expression func = rewrite_car (p);
    Object proc = null;
    ReferenceExp ref = null;

    if (func instanceof ReferenceExp)
      {
	ref = (ReferenceExp) func;
        Declaration decl = ref.getBinding();
	if (decl == null)
	  {
            String name = ref.getName();
            Binding binding = env.lookup((String) name);
	    if (binding != null && binding.isBound())
              proc = binding.get();
	    if (proc instanceof Syntax)
	      return apply_rewrite ((Syntax) proc, p);
            if (proc instanceof AutoloadProcedure)
              {
                try
                  {
                    proc = ((AutoloadProcedure) proc).getLoaded();
                  }
                catch (RuntimeException ex)
                  {
                    proc = null;
                  }
              }
	    if (proc instanceof Inlineable)
	      func = new QuoteExp(proc);
	  }
        else if (decl instanceof Syntax)
          return apply_rewrite ((Syntax) decl, p);
      }

    int cdr_length = List.length (cdr);

    if (func instanceof QuoteExp)
      {
	proc = ((QuoteExp) func).getValue();
	if (proc instanceof Procedure)
	  {
	    String msg = WrongArguments.checkArgCount((Procedure) proc,
						      cdr_length);
	    if (msg != null)
	      return syntaxError(msg);
	  }
      }

    Expression[] args = new Expression[cdr_length];

    for (int i = 0; i < cdr_length; i++)
      {
	Pair cdr_pair = (Pair) cdr;
	args[i] = rewrite_car (cdr_pair);
	cdr = cdr_pair.cdr;
      }

  tryDirectCall:
    if (proc != null && proc instanceof Procedure)
      {
	if (proc instanceof AutoloadProcedure)
	  {
	    try
	      {
		proc = ((AutoloadProcedure) proc).getLoaded();
	      }
	    catch (RuntimeException ex)
	      {
		break tryDirectCall;
	      }
	    if (proc == null || ! (proc instanceof Procedure))
	      break tryDirectCall;
	  }
	Class procClass = proc.getClass();
	if (procClass.getClassLoader() != systemClassLoader)
	  break tryDirectCall;
	try
	  {
	    java.lang.reflect.Method[] meths = procClass.getDeclaredMethods();
	    java.lang.reflect.Method best = null;
	    Class[] bestTypes = null;
            Procedure pproc = (Procedure) proc;
	    String name = pproc.getName();
	    if (name == null)
	      break tryDirectCall;
	    String mangledName = Compilation.mangleName(name);
	    for (int i = meths.length;  --i >= 0; )
	      {
		java.lang.reflect.Method meth = meths[i];
		int mods = meth.getModifiers();
		if ((mods & (Modifier.STATIC|Modifier.PUBLIC))
		    != (Modifier.STATIC|Modifier.PUBLIC))
		  continue;
		String mname = meth.getName();
		if (! mname.equals("apply") && ! mname.equals(mangledName))
		  continue;
		Class[] ptypes = meth.getParameterTypes();
		if (ptypes.length != cdr_length)
		  continue;
		// In the future, we may try to find the "best" match.
		if (best != null)
		  return syntaxError("ambiguous inline for call to "+name
				     + " (in class "+procClass.getName()+")");
		best = meth;
		bestTypes = ptypes;
	      }
	    if (best != null)
	      {
		ClassType procType = ClassType.make(procClass.getName());
		Type[] argTypes = new Type[bestTypes.length];
		for (int i = argTypes.length;  --i >= 0; )
		  argTypes[i] = Type.make(bestTypes[i]);
		gnu.bytecode.Method method
		  = procType.addMethod(best.getName(), best.getModifiers(),
				       argTypes,
				       Type.make(best.getReturnType()));
                pproc = new PrimProcedure(method);
                pproc.setName(name);
		func = new QuoteExp(pproc);
	      }
	  }
	catch (SecurityException ex)
	  {
	  }
      }

    return new ApplyExp (func, args);
  }

  /**
   * Re-write a Scheme expression in S-expression format into internal form.
   */
  public Expression rewrite (Object exp)
  {
    if (exp instanceof PairWithPosition)
      return rewrite_with_position (exp, (PairWithPosition) exp);
    else if (exp instanceof Pair)
      return rewrite_pair ((Pair) exp);
    else if (exp instanceof String)
      {
	String name = (String) exp;
	Object binding = current_decls.get (name);
	// Hygenic macro expansion may bind a renamed (uninterned) symbol
	// to the original symbol.  Here, use the original symbol.
	if (binding != null && binding instanceof String)
	  return new ReferenceExp ((String) binding);
	Declaration decl = (Declaration) binding;
	if (! isLexical(decl))
	  decl = null;
	return new ReferenceExp (name, decl);
      }
    else if (exp instanceof Expression)
      return (Expression) exp;
    else
      return new QuoteExp (exp);
  }

  public Expression rewrite_with_position (Object exp, PairWithPosition pair)
  {
    String save_filename = current_filename;
    int save_line = current_line;
    int save_column = current_column;
    Expression result;
    try
      {
	String exp_file = pair.getFile ();
	int exp_line = pair.getLine ();
	int exp_column = pair.getColumn ();
	current_filename = exp_file;
        current_line = exp_line;
	current_column = exp_column;
	if (exp == pair)
	  result = rewrite_pair (pair);  // To avoid a cycle
	else
	  result = rewrite (exp);
	if (result.getFile () == null)
	  result.setFile (exp_file);
	if (result.getLine () == 0)
	  result.setLine (exp_line, exp_column);
      }
    finally
      {
	current_filename = save_filename;
	current_line = save_line;
	current_column = save_column;
      }
    return result;
  }

  boolean scan_form (Object st, java.util.Vector forms, ScopeExp defs)
  {
    for (;;)
      {
        // Process st.
        if (! (st instanceof Pair))
          forms.addElement (st);
        else
          {
            Pair st_pair = (Pair) st;
            Object op = st_pair.car;
            Syntax syntax = check_if_Syntax (op);

            if (syntax != null && syntax instanceof Macro)
              {
                String save_filename = current_filename;
                int save_line = current_line;
                int save_column = current_column;
                Syntax saveSyntax = currentSyntax;
                try
                  {
                    if (st_pair instanceof PairWithPosition)
                      {
                        PairWithPosition ppair = (PairWithPosition) st_pair;
                        current_filename = ppair.getFile ();
                        current_line = ppair.getLine ();
                        current_column = ppair.getColumn ();
                      }
                    currentSyntax = syntax;
                    st = ((Macro) syntax).expand (st_pair, this);
                  }
                finally
                  {
                    current_filename = save_filename;
                    current_line = save_line;
                    current_column = save_column;
                    currentSyntax = saveSyntax;
                  }
                continue;
		}
            else if (syntax == Scheme.beginSyntax)
              {
                if (! scan_body (st_pair.cdr, forms, defs))
                  return false;
              }
            /*
            else if (syntax == Scheme.defineSyntax
                     && st_pair.cdr instanceof Pair
                     && ! (current_scope instanceof ModuleExp))
              {
                Object name = ((Pair) st_pair.cdr).car;
                if (name instanceof String)
                  defs.addDeclaration((String) name);
                else if (name instanceof Pair)
                  {
                    Pair name_pair = (Pair) name;
                    if (name_pair.car instanceof String)
                      defs.addDeclaration((String) name_pair.car);
                  }
                forms.addElement (st);
              }
            */
            else if ((syntax == Scheme.defineSyntax
                      || syntax == Scheme.defineSyntaxPrivate)
                     // Later:  || syntax == Scheme.defineSyntax
                     && st_pair.cdr instanceof Pair)
              {
                boolean makePrivate = syntax == Scheme.defineSyntaxPrivate;
                Pair p = (Pair) st_pair.cdr;
                Object name = p.car;
                Declaration decl = null;
                Pair declForm = null;
                if (name instanceof String)
                  {
                    decl = new Declaration((String) name);
                    declForm = makePair(p, decl, p.cdr);
                    st = makePair(st_pair, syntax, declForm);
                  }
                else if (name instanceof Pair)
                  {
                    Pair name_pair = (Pair) name;
                    if (name_pair.car instanceof String)
                      {
                        decl = new Declaration((String) name_pair.car);
                        declForm = makePair(name_pair, decl, name_pair.cdr);
                        p = makePair(p, declForm, p.cdr);
                        st = makePair(st_pair, syntax, p);
                      }
		    }
                if (decl != null)
                  {
                    if (defs instanceof ModuleExp)
                      {
                        mustCompileHere();
                        push(decl);
                        if (! makePrivate)
                          {
                            decl.setCanRead(true);
                            // decl.setCanWrite(true);
                          }
                      }
                    if (declForm instanceof PairWithPosition)
                      {
                        PairWithPosition declPos = (PairWithPosition) declForm;
                        decl.setFile(declPos.getFile());
                        decl.setLine(declPos.getLine(), declPos.getColumn());
                      }
                    defs.addDeclaration(decl);
                  }
                forms.addElement (st);
              }
            else if (syntax == Scheme.defineAliasSyntax
                     && st_pair.cdr instanceof Pair
                     && ! (current_scope instanceof ModuleExp)
                     &&  ((Pair) st_pair.cdr).car instanceof String)
              {
                Object name = ((Pair) st_pair.cdr).car;
                Type typeLocation = ClassType.make("gnu.mapping.Location");
                Declaration decl
                  = defs.addDeclaration((String) name, typeLocation);
                decl.setIndirectBinding(true);
                forms.addElement (st);
              }
            else if (syntax == Scheme.defineSyntaxSyntax
                     && st_pair.cdr instanceof Pair
                     && ! (current_scope instanceof ModuleExp)
                     &&  ((Pair) st_pair.cdr).car instanceof String)
              {
                Pair p = (Pair) st_pair.cdr;
                Object name = p.car;
                if (! (p.car instanceof String)
                    || ! (p.cdr instanceof Pair)
                    || (p = (Pair) p.cdr).cdr != List.Empty)
                  {
                    forms.addElement(syntaxError("invalid syntax for define-syntax"));
                    return false;
                  }
                Macro macro = new Macro((String) name, p.car);
                defs.addDeclaration(macro);
                p = makePair(st_pair, syntax, new Pair(macro, p));
                forms.addElement (p);
              }
            else
              forms.addElement (st);
          }
        return true;
      }
  }

  /** Recursive helper method for rewrite_body.
   * Scan body for definitions, placing partially macro-expanded
   * expressions into forms.
   * If definitions were seen, return a LetExp containing the definitions.
   */

  public boolean scan_body (Object body, java.util.Vector forms, ScopeExp defs)
  {
    while (body != List.Empty)
      {
	if (! (body instanceof Pair))
	  {
	    forms.addElement (syntaxError ("body is not a proper list"));
	    return false;
	  }
	Pair pair = (Pair) body;
	Object st = pair.car;
        if (! scan_form (st, forms, defs))
          return false;
	body = pair.cdr;
      }
    return true;
  }

  public static Pair makePair(Pair pair, Object car, Object cdr)
  {
    if (pair instanceof PairWithPosition)
      return new PairWithPosition((PairWithPosition) pair, car, cdr);
    return new Pair(car, cdr);
  }

  /**
   * Re-write a Scheme <body> in S-expression format into internal form.
   */

  public Expression rewrite_body (Object exp)
  {
    java.util.Vector forms = new java.util.Vector(20);
    LetExp defs = new LetExp(null);
    if (! scan_body (exp, forms, defs))
      return new ErrorExp("error while scanning in body");
    return rewrite_body(forms, defs);
  }

  public Expression rewrite_body (java.util.Vector forms, LetExp defs)
  {
    int nforms = forms.size();
    if (nforms == 0)
      return syntaxError ("body with no expressions");
    int ndecls = defs.countDecls();
    if (ndecls != 0)
      {
        Expression[] inits = new Expression[ndecls];
        for (int i = ndecls;  --i >= 0; )
          inits[i] = QuoteExp.nullExp;
        defs.inits = inits;
	push(defs);
      }
    Expression body;
    if (nforms == 1)
      body = rewrite (forms.elementAt(0));
    else
      {
	Expression[] exps = new Expression [nforms];
	for (int i = 0; i < nforms; i++)
	  exps[i] = rewrite (forms.elementAt(i));
	body = new BeginExp (exps);
      }
    if (ndecls == 0)
      return body;
    defs.body = body;
    pop(defs);
    return defs;
  }

  public void finishModule(ModuleExp mexp, java.util.Vector forms)
  {
    int nforms = forms.size();
    int ndecls = mexp.countDecls();
    pushDecls(mexp);
    Expression body;
    if (nforms == 1)
      body = rewrite(forms.elementAt(0));
    else
      {
	Expression[] exps = new Expression [nforms];
	for (int i = 0; i < nforms; i++)
	  exps[i] = rewrite(forms.elementAt(i));
	body = new BeginExp (exps);
      }
    mexp.body = body;
    pop(mexp);
    /* DEBUGGING:
    OutPort err = OutPort.errDefault ();
    err.print ("[Re-written expression for load/compile: ");
    mexp.print (err);
    //err.print ("\nbefore load<"+mod.getClass().getName()+">");
    err.println();
    err.flush();
    */
  }

  /**
   * Insert decl into current_decls.
   * (Used at rewrite time, not eval time.)
   */
  void push (Declaration decl)
  {
    String sym = decl.symbol();
    if (sym == null)
      return;
    Object old_decl = current_decls.get (sym);
    if (old_decl != null)
      decl.shadowed = old_decl;
    current_decls.put (sym, decl);
  }

  /** Remove this from Translator.current_decls.
   * (Used at rewrite time, not eval time.)
   */
  void pop (Declaration decl)
  {
    String sym = decl.symbol();
    if (sym == null)
      return;
    if (decl.shadowed == null)
      current_decls.remove (sym);
    else
      current_decls.put (sym, decl.shadowed);
  }

  public final void pushDecls (ScopeExp scope)
  {
    for (Variable var = scope.firstVar();  var != null;  var = var.nextVar())
      {
	// Don't make artificial variables visible.
	if (! var.isArtificial())
	  push((Declaration)var);
      }
  }

  public final void popDecls (ScopeExp scope)
  {
    for (Variable var = scope.firstVar();  var != null;  var = var.nextVar())
      {
	if (! var.isArtificial ())
	  pop((Declaration)var);
      }
  }

  public void push (ScopeExp scope)
  {
    scope.outer = current_scope;
    if (! (scope instanceof LambdaExp)) // which implies: outer != null
      mustCompileHere();
    current_scope = scope;
    pushDecls(scope);
  }

  public void pop (ScopeExp scope)
  {
    popDecls(scope);
    current_scope = scope.outer;
  }

}
