;; "anti-diff.scm" rational-function anti-derivative.	-*-scheme-*-
;; Copyright 2020 Aubrey Jaffer
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or (at
;; your option) any later version.
;;
;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

(require 'common-list-functions)

(define (poly:integrate p var)
  (define acc 0)
  (let lp ((coeffs (cdr (promote var p)))
	   (dg 0))
    (cond ((null? coeffs) (expr:normalize acc))
	  (else
	   (set! acc
		 (app* $1+$2
		       (app* $1/$2 (univ:demote
				    (univ:monomial (car coeffs) (+ dg 1) var))
			     (+ 1 dg))
			   acc))
	   (lp (cdr coeffs) (+ 1 dg))))))

;; returns a list of unsquared factors of increasing power.
(define (sqfr-splits c var)
  (define splitter (poly:diff c var))
  (cond ((not (number? splitter))
	 (let ((d '())
	       (aj '())
	       (bi (poly:gcd c splitter)))
	   (do ((b bi (poly:/ b d))
		(a (poly:/ c bi) d))
	       ((number? b)
		(reverse (cons a aj)))
	     (set! d (poly:gcd a b))
	     (set! aj (cons (poly:/ a d) aj)))))
	((number? c) (list c))
	(else (list (univ:norm0 var (cdr c))))))

;; The inverse of sqfr-splits.  Returns the product of increasing
;; powers of list of factors fcts.
(define (powfacts fcts)
  (let lp ((fcts fcts) (n 1) (acc 1))
    (cond ((null? fcts) acc)
	  (else
	   (lp (cdr fcts)
	       (+ 1 n)
	       (poly:* (ipow-by-squaring (car fcts) n 1 poly:*)
		       acc))))))

(define (rat:integrate nmf dnmf L var)
  (define N+ (reduce-init poly:* 1 (cdr nmf)))
  (define N- (reduce-init poly:* 1 (cdr dnmf)))
  (define M+ 0)
  (define M- 0)
  (define j 0)
  (if math:debug (math:print 'nmf= nmf 'dnmf= dnmf))
  (cond
   ((not (or (one? (car dnmf)) (univ:one? (car dnmf) var)))
    (math:warn 'denominator-has-residue (car dnmf)) 0)
   (else
    (for-each (lambda (p)
		(set! M+ (poly:+ (poly:*
				  (reduce-init poly:* 1 (butnth j nmf))
				  (poly:* (+ 1 j) (poly:diff p var)))
				 M+))
		(set! j (+ 1 j)))
	      nmf)
    (set! j 0)
    (for-each (lambda (p)
		(set! M- (poly:+ (poly:*
				  (reduce-init poly:* 1 (butnth j (cdr dnmf)))
				  (poly:* (- -1 j) (poly:diff p var)))
				 M-))
		(set! j (+ 1 j)))
	      (cdr dnmf))
    (let ((M (poly:+ (poly:* M+ N-) (poly:* M- N+)))
	  (N (poly:* N- N+)))
      (define A 0)
      (define R L)
      (define deg-M (poly:degree M var))
      (define deg-N (poly:degree N var))
      (define (gap deg-R) (min (- deg-R deg-M) (- deg-R deg-N -1)))
      (define (fnl) (app* $1*$2 A (app* $1/$2 (powfacts nmf) (powfacts (cdr dnmf)))))
      (if math:debug (math:print 'deg-M= deg-M 'M= M 'deg-N= deg-N 'N= N))
      (if (<= deg-N deg-M) (math:warn 'deg_N<=deg_M))
      (let lp ((deg-R (poly:degree (num R) var)))
	(define w (gap deg-R))
	(cond ((>= w 0)
	       (let ((lcR (poly:leading-coeff (num R) var)))
		 (define lcM (poly:coeff M var (- deg-R w)))
		 (define lcN (poly:coeff N var (- deg-R w -1)))
		 (define C (expr:normalize
			    (app* $1/$2
				  lcR
				  (poly:* (poly:+ lcM (poly:* w lcN))
					  (denom R)))))
		 (if math:debug (math:print 'w= w 'deg-R= deg-R 'R= R 'A= A 'C= C))
		 ;; (math:print 'deg-R= deg-R 'lcR= lcR 'R= R 'lcM= lcM 'lcN= lcN 'C= C)
		 (set! A (expr:normalize
			  (app* $1*$2+$3 C (univ:monomial 1 w var) A)))
		 (set! R (expr:normalize
			  (app* $1-$2*$3
				R
				C
				(poly:+
				 (poly:* M (univ:monomial 1 w var))
				 (poly:* N (univ:monomial w (- w 1) var))))))
		 (cond ((if (number? R) (zero? R) (univ:zero? R))
			(if math:debug (math:print 'A= A 'R= R)) ; 'w= (gap (poly:degree (num R) var))
			(fnl))
		       ((>= (gap (poly:degree (num R) var)) w)
			(math:warn 'non-decreasing-w 'w= (gap (poly:degree (num R) var)) 'A= A 'R= R)
			(fnl))
		       (else (lp (poly:degree (num R) var))))))
	      ((if (number? R) (zero? R) (univ:zero? R)) (fnl))
	      ((number? R) (math:warn 'non-zero-constant-part R) (fnl))
	      (else 0)))))))

(define (indef-integrate p v)
  (define nm (num p))
  (define dnm (denom p))
  (define cnm (univ:cont (promote v nm)))
  (define cdnm (univ:cont (promote v dnm)))
  (define nm/cnm (poly:/ nm cnm))
  (define dnm/cdnm (poly:/ dnm cdnm))
  (define intg
    (app* $1*$2
	  (cond ((number? dnm/cdnm)
		 (poly:integrate nm/cnm v))
		(else
		 (let ((nmf (sqfr-splits nm/cnm v))
		       (dnmf (sqfr-splits dnm/cdnm v)))
		   (rat:integrate (cons 1 (cdr nmf))
				  dnmf
				  (poly:/ nm/cnm (powfacts (cdr nmf)))
				  v))))
	  (app* $1/$2 cnm cdnm)))
  (cond ((eqv? 0 intg) (math:warn 'integration-failed) 0)
	;; (math:debug
	;;  (let ((chk (expr:normalize (app* $1-$2 (diff intg v) expr))))
	;;    (math:print 'result= intg 'chk= chk)
	;;    (if (not (eqv? 0 chk))
	;;        (math:warn 'diff-of-integral-mismatch 'result= intg 'difference= chk))
	;;    intg))
	(else intg)))

(define (integrate . args)
  (if (not (<= 2 (length args) 4)) (bltn:error 'integrate wna args))
  (let ((expr (normalize (car args)))
	(var (expl->var (cadr args)))
	(lo (if (null? (cddr args)) #f (caddr args)))
	(hi (and (= 4 (length args)) (cadddr args))))
    (cond ((= 2 (length args))
	   (indef-integrate expr var))
	  (else
	   (let ((sexp (sexp:alpha-convert (list (var:sexp var))
					   (cano->sexp expr horner))))
	     (define ifun (indef-integrate (sexp->math sexp) $1))
	     (cond ((case (length args)
	     	      ((3) (app* ifun lo))
	     	      ((4) (app* $1-$2 (app* ifun hi) (app* ifun lo)))))))))))

(defbltn 'integrate 2 #f integrate)
