/*  GFAX - Gnome fax application
 *  Copyright (C) 1999 George A. Farris
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include <config.h>
#include <gnome.h>
#include <glade/glade.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include "gfax.h"
#include "setup.h"
#include "messages.h"

extern Setup *prop;
extern MgettyConfigData *mgetty;


static gchar *bps[] = {"115200","57600","38400","19200","9600","4800","2400"};
static gchar *bpsSwitch[] = {"None","115200","57600","38400","19200","9600","4800","2400"};
static gchar *mtype[] = {"auto","cls2","c2.0"};
static gchar *carrier[] = {"Y","N"};
static gchar *mtc[] = {"Y","N"};

struct  Druid {
	/* mgetty druid widgets */
	GtkEntry *modem;
	GtkOptionMenu *retries;
	GtkOptionMenu *bps;
	GtkOptionMenu *cont;
	GtkOptionMenu *runQueue;
	GtkText *allow;
	GtkText *deny;
};

struct Druid *druid;

void cancel(GtkWidget *widget, gpointer data);
void finish(GtkWidget *widget, gpointer data);


/*=========================================================================
 * void mgetty_get_config(gboolean calledFromMain, Setup *prop, 
 *							MgettyConfigData *mgetty)
 *
 *  *prop      : struct that holds all the widget pointers - setup.h
 *  *megetty   : struct that holds all mgetty config data  - setup.h
 *	Called from: setup.c [get_config()]
 *  Returns    : void
 *
 *  Gets all the config parameters for mgetty
 *-----------------------------------------------------------------------*/
void mgetty_get_config(gboolean calledFromMain, Setup *prop, MgettyConfigData *mgetty)
{

	FILE *fin;
	gchar buf[128];
	struct stat statbuf;
	gchar *fname;
		
	GString *tmpstr;
	gboolean is_default;	/* Indicates if default item returned from config file */

	if ( !calledFromMain ) {

		/* OK lets go get the config data -----------------------------------*/
		tmpstr = g_string_new(NULL);

		/* lets see if we control the config file or not */
		g_string_sprintf(tmpstr, "=%s/%s=/Mgetty/DisabledConfig=TRUE", GFAX_DATADIR, "Config");
		mgetty->disabledConfig = gnome_config_get_bool_with_default(tmpstr->str, &is_default);

		if (mgetty->disabledConfig == TRUE) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
                                     (prop->mgettyConfigControlCheckButton), TRUE);
			/* all widgets in the table go !sensitive */
			gtk_widget_set_sensitive(GTK_WIDGET(prop->mgettyTable), FALSE);

		} else {

			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
                             (prop->mgettyConfigControlCheckButton), FALSE);

			/* if everything is ok we get to here and get the settings */

			g_string_sprintf(tmpstr, "=%s/%s=/Mgetty/=", GFAX_DATADIR, "Config");
			gnome_config_push_prefix(tmpstr->str);

			mgetty->configFile = g_strdup(gnome_config_get_string_with_default(
								"ConfigFile=/etc/mgetty+sendfax/sendfax.config", &is_default));
			mgetty->portSpeed = gnome_config_get_int_with_default("PortSpeed=2", &is_default);
            mgetty->portSpeedSwitch = gnome_config_get_int_with_default("SwitchPortSpeed=0", &is_default);
			mgetty->openDelay = gnome_config_get_int_with_default("OpenDelay=0", &is_default);
			mgetty->debugLevel = gnome_config_get_int_with_default("DebugLevel=3", &is_default);
			mgetty->ignoreCarrier = gnome_config_get_int_with_default("IgnoreCarrier=1", &is_default);
			mgetty->modemPort = g_strdup(gnome_config_get_string_with_default("FaxDevices=modem", &is_default));
			mgetty->maxTries = gnome_config_get_int_with_default("MaxTries=3", &is_default);
			mgetty->maxTriesContinue = gnome_config_get_int_with_default("maxTriesContinue=0", &is_default);
			mgetty->modemInit = g_strdup(gnome_config_get_string_with_default("ModemInit=ATZ", &is_default));
			mgetty->modemHandshake = g_strdup(gnome_config_get_string_with_default("ModemHandshake=", &is_default));
            mgetty->modemQuirks = g_strdup(gnome_config_get_string_with_default("ModemQuirks=", &is_default));
			mgetty->modemDialCommand = g_strdup(gnome_config_get_string_with_default("ModemDialCommand=ATDT", &is_default));
			mgetty->modemType = gnome_config_get_int_with_default("ModemType=0", &is_default);
			mgetty->runQueuePeriod = gnome_config_get_int_with_default("RunQueuePeriod=0", &is_default);
	
			gnome_config_pop_prefix();

	
            /* set the widgets with the retrieved values - never set a NULL value */
			if (mgetty->modemInit != NULL)
					gtk_entry_set_text(GTK_ENTRY(prop->mgettyModemInitEntry), mgetty->modemInit);
			if (mgetty->configFile != NULL)
					gtk_entry_set_text(GTK_ENTRY(prop->mgettySendfaxConfigFileEntry), mgetty->configFile);
			if (mgetty->modemPort != NULL)
					gtk_entry_set_text(GTK_ENTRY(prop->mgettyModemPortCombo), mgetty->modemPort);
	
			if (mgetty->modemHandshake != NULL)
            		gtk_entry_set_text(GTK_ENTRY(prop->mgettyModemHandshakeEntry), mgetty->modemHandshake);
			if (mgetty->modemQuirks != NULL)
					gtk_entry_set_text(GTK_ENTRY(prop->mgettyModemQuirksEntry), mgetty->modemQuirks);
			if (mgetty->modemDialCommand != NULL)
					gtk_entry_set_text(GTK_ENTRY(prop->mgettyDialCommandEntry), mgetty->modemDialCommand);
	

			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgRunQueueOptionmenu),
							mgetty->runQueuePeriod);
			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyMaxTriesOptionmenu),
            							mgetty->maxTries);
			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyIgnoreCarrierOptionmenu),
										mgetty->ignoreCarrier);
			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyModemSpeedOptionmenu),
										mgetty->portSpeed);
			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyModemTypeOptionmenu),
										mgetty->modemType);
	

			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyMaxTriesContOptionmenu),
            							mgetty->maxTriesContinue);
			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyModemSwitchSpeedOptionmenu),
										mgetty->portSpeedSwitch);
			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyOpenDelayOptionmenu),
										mgetty->openDelay);
			gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyDebugLevelOptionmenu),
										mgetty->debugLevel);

			/* lets read the fax.allow file */
            fname = g_strdup_printf("%s/%s", g_dirname(mgetty->configFile), "fax.allow");
			if ((stat(fname, &statbuf) == 0)) {
				if ((fin = fopen(fname, "r")) != NULL) {
					while (fgets(buf, 128, fin) != NULL) {
						gtk_text_insert(GTK_TEXT(prop->mgUserAllowText), NULL, NULL, NULL,
										buf, -1);

					}
				}
				fclose(fin);
				g_free(fname);
			}

			/* lets read the fax.deny file */
            fname = g_strdup_printf("%s/%s", g_dirname(mgetty->configFile), "fax.deny");
			if ((stat(fname, &statbuf) == 0)) {
				if ((fin = fopen(fname, "r")) != NULL) {
					while (fgets(buf, 128, fin) != NULL) {
						gtk_text_insert(GTK_TEXT(prop->mgUserDenyText), NULL, NULL, NULL,
												   buf, -1);
					}
				}
				fclose(fin);
				g_free(fname);
			}

		}

		gtk_widget_draw(GTK_WIDGET(prop->setup), NULL);

		g_string_free(tmpstr, TRUE);
	}
	
	return;
}



/*=========================================================================
 * mgetty_set_config(Setup *prop, MgettyConfigData *mgetty)
 *
 *  *prop      : struct that holds all the widget pointers - setup.h
 *  *megetty   : struct that holds all mgetty config data  - setup.h
 *  Called from: setup.c [set_config()]
 *  Returns    : void
 *
 *	write the mgetty config data out.
 *
 *-----------------------------------------------------------------------*/
 
void mgetty_set_config(Setup *prop, MgettyConfigData *mgetty)
{
	FILE *fout;
	gchar *buf;
	gchar *fname;
	
	GString *tmpstr;
	GtkWidget *active_widget;

	tmpstr = g_string_new(NULL);
 		

	/* if we're not allowed to write to sendfax config file then there
	   is no point doing any of this */
	if (mgetty->disabledConfig == FALSE) {
		
		FILE *fp;
							
		mgetty->modemInit = g_strdup(gtk_entry_get_text((GtkEntry *)prop->mgettyModemInitEntry));
		mgetty->modemHandshake = g_strdup(gtk_entry_get_text((GtkEntry *)prop->mgettyModemHandshakeEntry));
		mgetty->modemQuirks = g_strdup(gtk_entry_get_text((GtkEntry *)prop->mgettyModemQuirksEntry));
		mgetty->modemDialCommand = g_strdup(gtk_entry_get_text((GtkEntry *)prop->mgettyDialCommandEntry));
		mgetty->modemPort = g_strdup(gtk_entry_get_text(prop->mgettyModemPortCombo));

		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgRunQueueOptionmenu->menu));
		mgetty->runQueuePeriod = g_list_index (GTK_MENU_SHELL(prop->mgRunQueueOptionmenu->menu)->children, 
					active_widget);

		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgettyModemSwitchSpeedOptionmenu->menu));
		mgetty->portSpeedSwitch = g_list_index (GTK_MENU_SHELL(prop->mgettyModemSwitchSpeedOptionmenu->menu)->children, 
					active_widget);

		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgettyOpenDelayOptionmenu->menu));
		mgetty->openDelay = g_list_index (GTK_MENU_SHELL(prop->mgettyOpenDelayOptionmenu->menu)->children, 
					active_widget);

		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgettyDebugLevelOptionmenu->menu));
		mgetty->debugLevel = g_list_index (GTK_MENU_SHELL(prop->mgettyDebugLevelOptionmenu->menu)->children, 
					active_widget);

		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgettyMaxTriesContOptionmenu->menu));
		mgetty->maxTriesContinue = g_list_index (GTK_MENU_SHELL(prop->mgettyMaxTriesContOptionmenu->menu)->children, 
					active_widget);
					
		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgettyMaxTriesOptionmenu->menu));
		mgetty->maxTries = g_list_index (GTK_MENU_SHELL(prop->mgettyMaxTriesOptionmenu->menu)->children, 
					active_widget);
	
		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgettyIgnoreCarrierOptionmenu->menu));
		mgetty->ignoreCarrier = g_list_index (GTK_MENU_SHELL(prop->mgettyIgnoreCarrierOptionmenu->menu)->children,
					active_widget);
	
		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgettyModemSpeedOptionmenu->menu));
		mgetty->portSpeed = g_list_index (GTK_MENU_SHELL(prop->mgettyModemSpeedOptionmenu->menu)->children,
					active_widget);

		active_widget = gtk_menu_get_active(GTK_MENU(prop->mgettyModemTypeOptionmenu->menu));
		mgetty->modemType = g_list_index (GTK_MENU_SHELL(prop->mgettyModemTypeOptionmenu->menu)->children,
					active_widget);
	
		mgetty->configFile = g_strdup(gtk_entry_get_text(
					(GtkEntry *)prop->mgettySendfaxConfigFileEntry));
	

		/* if everything is ok we get to here and save the settings */
		/* remember we must be root to write this */
 		g_string_sprintf(tmpstr, "=%s/%s=/Mgetty/=", GFAX_DATADIR, "Config");
		gnome_config_push_prefix(tmpstr->str);

		gnome_config_set_int("SwitchPortSpeed", mgetty->portSpeedSwitch);
		gnome_config_set_int("OpenDelay", mgetty->openDelay);
		gnome_config_set_int("DebugLevel", mgetty->debugLevel);
		gnome_config_set_int("maxTriesContinue", mgetty->maxTriesContinue);
		gnome_config_set_string("ConfigFile", mgetty->configFile);
		gnome_config_set_int("PortSpeed", mgetty->portSpeed);
		gnome_config_set_int("IgnoreCarrier", mgetty->ignoreCarrier);
		gnome_config_set_string("FaxDevices", mgetty->modemPort);
		gnome_config_set_int("MaxTries", mgetty->maxTries);
		gnome_config_set_string("ModemInit", mgetty->modemInit);
		gnome_config_set_string("ModemHandshake", mgetty->modemHandshake);
		gnome_config_set_string("ModemQuirks", mgetty->modemQuirks);
		gnome_config_set_string("ModemDialCommand", mgetty->modemDialCommand);
		gnome_config_set_int("ModemType", mgetty->modemType);
		gnome_config_set_int("RunQueuePeriod", mgetty->runQueuePeriod);

		gnome_config_pop_prefix();

		/* right now lets write the sendfax file. gota be root to do this */
		if ((fp = fopen(mgetty->configFile, "w")) == NULL) {
			do_message(_("\nI can't open the sendfax config file.  You must be\n \
							logged in as root to write this file.\n"));
							
			/* at least be smart enough to save all those hard earned settings */
			gnome_config_sync();	/* write to file */
			return;
		}

		
		/* we do some pretty simple checks here, afterall we don't have a modem 
		   database on hand */
		fprintf(fp, mgHeader);
		if (strlen(mgetty->modemInit) > 0) {
			fprintf(fp, mgModemInitHelp);
			fprintf(fp, "modem-init %s\n", mgetty->modemInit);
		}
		if (strlen(mgetty->modemHandshake) > 0) {
			fprintf(fp, mgModemHandshakeHelp);
			fprintf(fp, "modem-handshake %s\n", mgetty->modemHandshake);
		}
		if (strlen(mgetty->modemQuirks) > 0) {
			fprintf(fp, mgModemQuirksHelp);
			fprintf(fp, "modem-quirks %s\n", mgetty->modemQuirks);			
		}
		if (strlen(mgetty->modemPort) < 1) {
			do_message(_("\nYou must choose a serial port!\n \
							That the modem is connected to.\n"));
			/* at least be smart enough to save all those hard earned settings */
			gnome_config_sync();	/* write to file */
			return;
		} else {
			fprintf(fp, mgModemPortHelp);
			fprintf(fp, "fax-devices %s\n", mgetty->modemPort);
		}
		fprintf(fp, mgPortSpeedHelp);
		fprintf(fp, "speed %s\n", bps[mgetty->portSpeed]);

		if (mgetty->portSpeedSwitch > 0) {
			fprintf(fp, mgSwitchBdHelp);
			fprintf(fp, "switchbd %s\n", bpsSwitch[mgetty->portSpeedSwitch]);			
		}
		fprintf(fp, mgModemTypeHelp);
		fprintf(fp, "modem-type %s\n", mtype[mgetty->modemType]);
		fprintf(fp, mgIgnoreCarrierHelp);
		fprintf(fp, "ignore-carrier %s\n", carrier[mgetty->ignoreCarrier]);
		fprintf(fp, mgMaxTriesHelp);
		fprintf(fp, "max-tries %d\n", mgetty->maxTries);
		fprintf(fp, mgMaxTriesContHelp);
		fprintf(fp, "max-tries-continue %s\n", mtc[mgetty->maxTriesContinue]);			
		if (strlen(mgetty->modemDialCommand) > 0) {
			fprintf(fp, mgDialCommandHelp);			
			fprintf(fp, "dial-prefix %s\n", mgetty->modemDialCommand);
		}
		fprintf(fp, mgDebugLevelHelp);
		fprintf(fp, "debug-level %d\n", mgetty->debugLevel);			
		if (mgetty->openDelay > 0) {
			fprintf(fp, mgOpenDelayHelp);
			fprintf(fp, "open-delay %d\n", mgetty->openDelay * 500);			
		}
		fclose(fp);

		switch(mgetty->runQueuePeriod) {

			case 0:	/* delete it */
					append_crontab_entry(NULL);
					break;
			case 1:	/* every minute */
					append_crontab_entry("* * * * * root /usr/bin/faxrunq\n");
					break;
			case 2: /* every 5 minutes */
					append_crontab_entry("*/5 * * * * root /usr/bin/faxrunq\n");
					break;
			case 3: /* every 10 minutes */
					append_crontab_entry("*/10 * * * * root /usr/bin/faxrunq\n");
					break;
			case 4: /* every 15 minutes */
					append_crontab_entry("*/15 * * * * root /usr/bin/faxrunq\n");
					break;
			case 5: /* every 30 minutes */
					append_crontab_entry("*/30 * * * * root /usr/bin/faxrunq\n");
					break;
			case 6: /* every hour */
					append_crontab_entry("1 * * * * root /usr/bin/faxrunq\n");
					break;

		}

		/* lets write the fax.allow file */
		if (gtk_text_get_length(GTK_TEXT(prop->mgUserAllowText)) >= 2) {  /* allow for whitespace */
			fname = g_strdup_printf("%s/%s", g_dirname(mgetty->configFile), "fax.allow");
			buf = g_strdup(gtk_editable_get_chars(GTK_EDITABLE(prop->mgUserAllowText), 0, -1));
		
	  		if ((fout = fopen(fname, "w")) != NULL) {
				fputs(buf, fout);
			}
			fclose(fout);
			g_free(buf);
			g_free(fname);
		}

		/* lets write the fax.deny file */
		if (gtk_text_get_length(GTK_TEXT(prop->mgUserDenyText)) >= 2) {			
			fname = g_strdup_printf("%s/%s", g_dirname(mgetty->configFile), "fax.deny");
			buf = g_strdup(gtk_editable_get_chars(GTK_EDITABLE(prop->mgUserDenyText), 0, -1));

	  		if ((fout = fopen(fname, "w")) != NULL) {
				fputs(buf, fout);
			}
			fclose(fout);
			g_free(buf);
			g_free(fname);
		}
		
	} /* end mgetty->disabledConfig */

	g_string_free(tmpstr, TRUE);

	return;
}


/*=========================================================================
 * mgetty_setup_druid(GtkWidget *widget, gpointer data)
 *
 *  *widget    : widget pointer from druid button
 *  *data      : not used
 *  Called from: [glade] "Setup Druid" button
 *  Returns    : void
 *
 *	Will eventually be the druid code.
 *
 *-----------------------------------------------------------------------*/
void mgetty_setup_druid(GtkWidget *widget, gpointer data)
{
	GladeXML *window;	/* The main window */
	FILE *fin;
	gchar buf[128];
	struct stat statbuf;
	gchar *fname;
	gint len;

 	if (g_strcasecmp(g_get_user_name(), "root") != 0) {	
        do_message(_("\nYou must be \"root\" to change mgetty settings!\n"));
        return;
	}

	druid = (struct Druid*)malloc(sizeof(struct Druid));
	
	window = glade_xml_new(GLADEDIR "/mgettydruid.glade", "mgettyDruidDialog");

	/* connect the signals */		
	glade_xml_signal_autoconnect(window);
	

	/* get the widgets */
	druid->modem = (GtkEntry *)glade_xml_get_widget(window, "modemComboEntry");
	druid->bps = (GtkOptionMenu *)glade_xml_get_widget(window, "bpsOptionmenu");
	druid->retries = (GtkOptionMenu *)glade_xml_get_widget(window, "retryOptionmenu");
	druid->cont = (GtkOptionMenu *)glade_xml_get_widget(window, "contOptionmenu");
	druid->runQueue = (GtkOptionMenu *)glade_xml_get_widget(window, "cronOptionmenu");
	druid->allow = (GtkText *)glade_xml_get_widget(window, "allowText");
	druid->deny = (GtkText *)glade_xml_get_widget(window, "denyText");

	/* We at least should preserve this information if it's been set up before */
	/* lets read the fax.allow file */
    fname = g_strdup_printf("%s/%s", g_dirname(mgetty->configFile), "fax.allow");
	if ((stat(fname, &statbuf) == 0)) {
		if ((fin = fopen(fname, "r")) != NULL) {
			while (fgets(buf, 128, fin) != NULL) {
				gtk_text_insert(GTK_TEXT(druid->allow), NULL, NULL, NULL,
								buf, -1);

			}
		}
		fclose(fin);
		g_free(fname);
	}

	/* lets read the fax.deny file */
    fname = g_strdup_printf("%s/%s", g_dirname(mgetty->configFile), "fax.deny");
	if ((stat(fname, &statbuf) == 0)) {
		if ((fin = fopen(fname, "r")) != NULL) {
			while (fgets(buf, 128, fin) != NULL) {
				/* if it already has entries clear "root" which is put in by default */
				if (strlen(buf) > 1 ) {
					len = gtk_text_get_length(GTK_TEXT(druid->deny)); 	/* get text length */
					gtk_text_set_point(GTK_TEXT(druid->deny), 0);		/* point to beginning */
					gtk_text_forward_delete(GTK_TEXT(druid->deny), len);	/* delete all chars */
				}
				gtk_text_insert(GTK_TEXT(druid->deny), NULL, NULL, NULL,
										   buf, -1);
			}
		}
		fclose(fin);
		g_free(fname);
	}

	return;
}


void cancel(GtkWidget *widget, gpointer data)
{
	gtk_object_destroy((GtkObject *)gtk_widget_get_toplevel(widget));
}


void finish(GtkWidget *widget, gpointer data)
{
	gchar *buf;
	gint len;
	GtkWidget *active_widget;
	
	
	/* set the right value in the mgetty structure then update the widget in the main
	   setup window.  This way we can just call mgetty_setup and everything will
	   have the correct values */
	mgetty->modemPort = g_strdup(gtk_entry_get_text(druid->modem));
	gtk_entry_set_text(GTK_ENTRY(prop->mgettyModemPortCombo), mgetty->modemPort);
	
	active_widget = gtk_menu_get_active(GTK_MENU(druid->bps->menu));
	mgetty->portSpeed = g_list_index (GTK_MENU_SHELL(druid->bps->menu)->children, 
					active_widget);
	gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyModemSpeedOptionmenu),
					mgetty->portSpeed);

	active_widget = gtk_menu_get_active(GTK_MENU(druid->cont->menu));
	mgetty->maxTriesContinue = g_list_index (GTK_MENU_SHELL(druid->cont->menu)->children, 
					active_widget);
	gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyMaxTriesContOptionmenu), 
					mgetty->maxTriesContinue);
					
	active_widget = gtk_menu_get_active(GTK_MENU(druid->retries->menu));
	mgetty->maxTries = g_list_index (GTK_MENU_SHELL(druid->retries->menu)->children, 
					active_widget);
	gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgettyMaxTriesOptionmenu), 
					mgetty->maxTries);
														
	active_widget = gtk_menu_get_active(GTK_MENU(druid->runQueue->menu));
	mgetty->runQueuePeriod = g_list_index (GTK_MENU_SHELL(druid->runQueue->menu)->children, 
					active_widget);
	gtk_option_menu_set_history(GTK_OPTION_MENU(prop->mgRunQueueOptionmenu), 
					mgetty->runQueuePeriod);

	buf = g_strdup(gtk_editable_get_chars(GTK_EDITABLE(druid->allow), 0, -1));
	len = gtk_text_get_length(GTK_TEXT(prop->mgUserAllowText)); 	/* get text length */
	gtk_text_set_point(GTK_TEXT(prop->mgUserAllowText), 0);			/* point to beginning */
	gtk_text_forward_delete(GTK_TEXT(prop->mgUserAllowText), len);	/* delete all chars */
	gtk_text_insert(GTK_TEXT(prop->mgUserAllowText), NULL, NULL, NULL, buf, -1);
	g_free(buf);

	buf = g_strdup(gtk_editable_get_chars(GTK_EDITABLE(druid->deny), 0, -1));
	len = gtk_text_get_length(GTK_TEXT(prop->mgUserDenyText));
	gtk_text_set_point(GTK_TEXT(prop->mgUserDenyText), 0);
	gtk_text_forward_delete(GTK_TEXT(prop->mgUserDenyText), len);
	gtk_text_insert(GTK_TEXT(prop->mgUserDenyText), NULL, NULL, NULL, buf, -1);	
	g_free(buf);
	
	gtk_widget_draw(GTK_WIDGET(prop->setup), NULL);
	
	gtk_object_destroy((GtkObject *)gtk_widget_get_toplevel(widget));
	
}


/*=========================================================================
 * append_crontab_entry(gchar *fields)
 *
 *  *fields    : fields of a crontab entry
 *  Called from: mgetty_set_config() - this file
 *  Returns    : void
 *
 *	Appends a crontab entry to /etc/crontab
 *
 *-----------------------------------------------------------------------*/
void append_crontab_entry(gchar *fields)
{
 	FILE *fout;
	
	if (fields == NULL) {
		delete_crontab_entry("faxrunq");	
	
	} else {
		
		/* add the field to the crontab file */
		delete_crontab_entry("faxrunq");	/* first we delete the entry thats there */
		
		if ((fout = fopen("/etc/crontab", "a")) == NULL) {
			do_message(_("\nCan't open /etc/crontab\n"));
			return;
		}
		fputs(fields, fout);
		fclose(fout);
	}
}


/*=========================================================================
 * delete_crontab_entry(gchar *entry)
 *
 *  *entry     : crontab line
 *  Called from: mgetty_set_config() - this file
 *  Returns    : void
 *
 *	Deletes a crontab entry from /etc/crontab
 *
 *-----------------------------------------------------------------------*/
void delete_crontab_entry(gchar *entry)
{
 	FILE *fin, *fout;
	gchar buf[128];
	gchar c;


	/* if it's NULL then nuke the faxrunq crontab entry.  */
 	if ((fin = fopen("/etc/crontab", "r")) == NULL) {
		do_message(_("\nCan't open /etc/crontab\n"));
		return;
	}
  	if ((fout = fopen("/tmp/crontab", "w")) == NULL) {
		do_message(_("\nCan't create /tmp/crontab\n"));
		return;
	}
	
	while (fgets(buf, 128, fin) != NULL) {
		if (strstr(buf, entry)) {
			continue;
		} else {
			fputs(buf, fout);
		}
	}
	fclose(fout);
	fclose(fin);

	/* now copy the new file back to /etc */
  	if ((fin = fopen("/tmp/crontab", "r")) == NULL) {
		do_message(_("\nCan't open /tmp/crontab\n"));
		return;
	}
  	if ((fout = fopen("/etc/crontab", "w")) == NULL) {
		do_message(_("\nCan't create /etc/crontab\n"));
		return;
	}

	while ((c = fgetc(fin)) != EOF)
		fputc(c, fout);

	fclose(fout);
	fclose(fin);
	remove("/tmp/crontab");
}

/* These are Emacs variables to use a common "Gfax" coding style:
 * ;;; Local Variables: ***
 * ;;; mode:C ***
 * ;;; c-basic-offset:8 ***
 * ;;; indent-tabs-mode:nil ***
 * ;;; End: ***
 */

