# Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016,
#   2017, 2018, 2019, 2020,
#   2021, 2022, 2023, 2024 Free Software Foundation, Inc.

# This file is part of GNUnited Nations.

# GNUnited Nations is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.

# GNUnited Nations is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with GNUnited Nations.  If not, see <http://www.gnu.org/licenses/>.

# SYNOPSIS
#
#   AX_PO4A()
#
# DESCRIPTION
#
# Configure PO4A executables to use.
# In particular, test if po4a-updatepo is usable for making POT files.
# Set PO4A, PO4A_GETTEXTIZE, PO4A_TRANSLATE PO4A_UPDATE, have_po4a,
# enable_gnun_po4a_translate, enable_gnun_po4a_updatepo variables.
# Define HAVE_PO4A conditional.

AC_DEFUN([AX_PO4A], [
  AX_PO4A_PROGS
  AX_PO4A_CONFTEST
  have_po4a_prog=no
  po4a_head=$PO4A_UPDATEPO
  po4a_tail="$po4a_args"
  AX_HAVE_PO4A_PROG([have_po4a_tepo])
  AS_CASE([$have_po4a_tepo], [no], [PO4A_UPDATEPO=])
  po4a_head=$PO4A_GETTEXTIZE
  AX_HAVE_PO4A_PROG([have_po4a_ize])
  had_po4a_prog=$have_po4a_prog
  po4a_head=$PO4A
  po4a_tail="--no-translations $po4a_common_args conftest.po4a"
  AX_HAVE_PO4A_PROG([have_po4a_wrapped])
  AS_CASE([$have_po4a_wrapped-$enable_gnun_po4a_updatepo], [yes-yes], [
    # Use GNUN wrapper as po4a-updatepo per the configure option.
    had_po4a_prog=no
  ])
  AS_CASE([$have_po4a_wrapped-$had_po4a_prog], [yes-no], [
    PO4A_UPDATEPO="$bindir/gnun-po4a-updatepo"
  ])
  AS_CASE([$have_po4a_prog], [no], [have_po4a=no])
  # PO4A is useless for our purpose if awk is missing.
  AS_CASE([$decent_awk], [no], [have_po4a=no])
  AX_POREFS
  AM_CONDITIONAL([HAVE_PO4A], [test "$have_po4a" = yes])
]) dnl AX_PO4A

# Run PO source reference-related tests against the given porefs argument.
AC_DEFUN([AX_POREF_CASE], [
  porefs_arg="$2"
  po4a_head=$po4a_prog
  po4a_tail="$porefs_arg $po4a_args"
  AX_HAVE_PO4A_PROG([accept_$1], [$po4a_prog accepts $porefs_arg])
  AS_CASE([$accept_$1], [yes], [
    AC_MSG_CHECKING([whether $porefs_arg really removes references])
    have_$1=no
    AS_IF([$GREP -q '^#: ' conftest.pot], [result=no], [
      result=yes
      AS_CASE(["$porefs_param"], ["not found"], [
        have_$1=yes
        porefs_param="$porefs_arg"
      ])
    ])
    AC_MSG_RESULT([$result])
  ])
]) dnl AX_POREF_CASE

# Configure PO source reference-related features.
AC_DEFUN([AX_POREFS], [
  porefs_param="not found"
  AS_CASE([$have_po4a], [yes], [
    AC_MSG_CHECKING([for the PO4A executable to create POT files])
    AC_MSG_RESULT([$po4a_prog])
    po4a_head=$po4a_prog
    AX_POREF_CASE([porefs_older], [-o porefs=none])
    AX_POREF_CASE([porefs_newer], [--porefs=never])
    AS_CASE(["$porefs_param"], ["not found"], [], [
      AC_SUBST([DROP_POREFS], [$porefs_param])
    ])
  ])
  AM_CONDITIONAL([HAVE_DROP_POREFS], [test "$porefs_param" != "not found"])
  AS_CASE(["$porefs_param-$have_po4a_wrapped"], ["not found-yes"], [
    PO4A_UPDATEPO="$bindir/gnun-po4a-updatepo"
    porefs_param=wrapped
  ])
]) dnl AX_POREFS

# Define a configure option for one of the two wrappers.
AC_DEFUN([AX_PO4A_ENABLE_WRAPPER], [
  AC_MSG_CHECKING([whether to use gnun-po4a-$1])
  AC_ARG_ENABLE([gnun-po4a-$1],
    [AS_HELP_STRING([--enable-gnun-po4a-$1],
    [Use GNUN wrapper against po4a instead of po4a-$1.])]
  )
  AS_CASE(["x$enable_gnun_po4a_$1"], [xyes], , [enable_gnun_po4a_$1=no])
  AC_MSG_RESULT([$enable_gnun_po4a_$1])
])

# Look for PO4A programs and related configure options.
AC_DEFUN([AX_PO4A_PROGS], [
  AX_ARG_PROG([PO4A], [po4a])
  AX_PO4A_VERSION
  AX_ARG_PROG([PO4A_GETTEXTIZE], [po4a-gettextize])
  AX_ARG_PROG([PO4A_UPDATEPO], [po4a-updatepo])
  AX_ARG_PROG([PO4A_TRANSLATE], [po4a-translate])
  AX_PO4A_ENABLE_WRAPPER([translate])
  AS_CASE(["x$enable_gnun_po4a_translate"],[xyes],
    [PO4A_TRANSLATE="$bindir/gnun-po4a-translate"])
  AX_PO4A_ENABLE_WRAPPER([updatepo])
]) dnl AX_PO4A_PROGS

# Check for minimum PO4A version.
AC_DEFUN([AX_PO4A_VERSION], [
  AC_CACHE_CHECK([if PO4A version is at least 0.37], [gnun_cv_recent_po4a],
    [gnun_cv_recent_po4a=no
    po4a_version=`
      $PO4A --version 2> /dev/null | $SED '/^po4a/!d;s/.$//;s/.* //;q'`
    AX_COMPARE_VERSION([$po4a_version], [ge], [0.37],
      [gnun_cv_recent_po4a=yes])])
  have_po4a=$gnun_cv_recent_po4a
])

# Check if a PO4A script defined by $po4a_head and $po4a_tail works,
# update the have_po4a_prog, po4a_prog and po4a_params variables unless
# a working PO4A script has already been found.
AC_DEFUN([AX_HAVE_PO4A_PROG], [
  AS_CASE(["x$2"], [x], [msg="$po4a_head can create POT files"], [msg="$2"])
  AC_MSG_CHECKING([whether $msg])
  rm -f conftest.pot
  result=no
  AS_IF([$po4a_head $po4a_tail > /dev/null 2>&1], [
    result=yes
    AS_CASE([$have_po4a_prog], [no], [
       have_po4a_prog=yes
       po4a_prog=$po4a_head
       po4a_params="$po4a_tail"
       PO4A_UPDATEPO=$po4a_prog
    ])
  ])
  AC_MSG_RESULT([$result])
  $1=$result
])

# SYNOPSIS
#
#   AX_PO4A_NOTICE()
#
# DESCRIPTION
#
# Interpret results of AX_PO4A and display the respective warnings.

AC_DEFUN([AX_PO4A_NOTICE], [
  AS_CASE(["$have_po4a-$po4a_version"], [no-0.7[[012]]-* | no-0.7[[01]]],
    [ AC_MSG_WARN([No working PO4A version is detected.
The detected PO4A version is $po4a_version.  GNUN input from www.gnu.org
is known to be incompatible with PO4A versions 0.70 and 0.71 when combined
with Perl versions older than 5.34.0.  POT and HTML generation is left
unconfigured.
])
  ]) dnl AS_CASE(["$have_po4a-$po4a_version"],..

  AS_CASE([$have_po4a_tepo], [no], [
    AS_CASE([$have_po4a_ize], [no], [
      AS_CASE([$have_po4a_wrapped], [no],
        [AC_MSG_WARN([No usable PO4A executables are found.
GNUN will not be able to generate POT and HTML files.
])
       ]
     )],
      [ AC_MSG_WARN([
The po4a-updatepo executable isn't usable, po4a-gettextize will be used.
])
      ])
  ]) dnl AS_CASE([$have_po4a_tepo]

  AS_CASE(["$have_po4a-$porefs_param"], ["yes-not found"], [
    AC_MSG_WARN([PO4A parameter for dropping source references not found.
POT files will be generated with references.
    ])], [yes-wrapped], [
    AC_MSG_WARN([PO4A parameter for dropping source references not found.
POT files will be post-processed in the PO4A wrapper.
    ])
  ]) dnl AS_CASE(["$porefs_param"], ["not found"],
]) dnl AX_PO4A_NOTICE

# Generate a test.  Sometimes Perl v5.32.1 or earlier with PO4A-0.70
# can't correctly compile POT files from non-ASCII HTMLs.
# This is a small test we could come up with.  NB the exact (length of the)
# path (./conftest.conf-test.proto) is significant.
AC_DEFUN([AX_PO4A_CONFTEST], [
  po4a_args="--master-charset=utf-8 -o ontagerror=silent"
  po4a_common_args="$po4a_args"
  po4a_args="$po4a_args --format=xhtml --master ./conftest.conf-test.proto"
  po4a_args="$po4a_args --po ./conftest.pot"
  AS_ECHO(['[[po4a_langs]] cv']) > ./conftest.po4a
  AS_ECHO(['[[po4a_paths]] conftest.pot cv:/dev/null']) >> ./conftest.po4a
  AS_ECHO(['[[type: xhtml]] ./conftest.conf-test.proto cv:/dev/null']) \
    >> ./conftest.po4a
  AS_ECHO(['<head><title>.. ... ... . .. .... .... ... ...... .. . . .
. .. .... . . . ... .. .. .. ... ... .... . .</title></head>
<body>
<p lang="aa">..</p>
<p lang="..">. .. .. .....</p>
<p lang="ab">... ..</p>
<p>..... .</p>
<p>... .. ... . . .... . . .. ... ...</p>
<blockquote>
<p>..... .... ....</p>
<p>...... ... ..</p>
<p>. .. .. .. .. ...... .. . .. .</p>
<p>.. .. ... .... .. .... .. . ... ...</p>
<p>স্বাধী... ..ফট...য়্যার</p>
</blockquote>
</body>']) > ./conftest.conf-test.proto
]) dnl AX_PO4A_CONFTEST
