/* 
 *    Programmed By: Mohammed Isam Mohammed [mohammed_isam1984@yahoo.com]
 *    Copyright 2014, 2015, 2016, 2017, 2018 (c)
 * 
 *    file: main.c
 *    This file is part of mino.
 *
 *    mino is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    mino is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with mino.  If not, see <http://www.gnu.org/licenses/>.
 */    
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <dirent.h>
#include <poll.h>
#include <signal.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include "defs.h"
#include "edit.h"
#include "file.h"
#include "kbd.h"
#include "options.h"
#include "modules/modules.h"

#define EXIT()		\
{			\
 restoreTerminal();	\
 fcloseall();		\
 exit(0);		\
}

extern char *STARTUP_FILE_NAME;     /* init.c */
extern void checkFileExtension();

void drawScrollBar();
int isKeyword(int pos, int start, int *wordlen);

static sig_atomic_t end = 0;
FILE *NULL_DEV;
char *open_file_name;

struct termios oldtio, curtio;
struct sigaction sa;

void confirmEOL(int pos)
{
    if(selectedChar >= MAX_CHARS_PER_LINE) selectedChar = MAX_CHARS_PER_LINE-1;
    if(selectedChar >= lines[pos]->charCount)
    {
        selectedChar = lines[pos]->charCount;
        if(lines[pos]->text[selectedChar-1] == '\n') selectedChar--;
    }
}

void gotoNextWord(int pos, int startAt)
{
    int i;
    char *s = lines[pos]->text+startAt;
    selectedChar = charsToBytes(pos, startAt);
    while(is_whitespace(*s)) s++, selectedChar++;
    if(*s == '\n' || *s == '\0') return;
    while(*s && !is_whitespace(*s))
    {
        if((*s & 0xc0) != 0x80) selectedChar++;
        s++;
    }
    if(*s == '\n' || *s == '\0') return;
    while(is_whitespace(*s)) s++, selectedChar++;
    /*
    for(int i = startAt; i < lines[pos]->charCount; i++)
    {
        char c1 = lines[pos]->text[i-1];
        char c2 = lines[pos]->text[i  ];
        char c3 = lines[pos]->text[i+1];
        if(is_whitespace(c2))
        {
            if(!is_whitespace(c3))
            {
                selectedChar = i+1; break;
            }
            else if(!is_whitespace(c1))
            {
                selectedChar = i; break;
            }
        }
    }
    */
}

void gotoPrevWord(int pos, int startAt)
{
    int i;
    char *s = lines[pos]->text;
    selectedChar = 0;
    while(is_whitespace(s[startAt])) startAt--;
    if(startAt <= 0) return;
    while(!is_whitespace(s[startAt])) startAt--;
    if(is_whitespace(s[startAt])) startAt++;
    if(startAt <= 0) return;
    for(i = 0; i < startAt; i++)
    {
        if((*s & 0xc0) != 0x80) selectedChar++;
    }
    /*
    for(int i = startAt; i >= 0; i--)
    {
        char c1 = lines[pos]->text[i-1];
        char c2 = lines[pos]->text[i  ];
        char c3 = lines[pos]->text[i+1];
        if(is_whitespace(c2))
        {
            if(!is_whitespace(c3))
            {
                selectedChar = i+1; break;
            }
            else if(!is_whitespace(c1))
            {
                selectedChar = i; break;
            }
        }
    }
    */
}

void sighandler(int signo)
{
    //fprintf(stdout, "SIGNAL %d received\n", signo);
    if(signo == 2) 
    {	//CTRL-C pressed
      editMenu_Copy();
    } 
    else 
    {
      end = 1;
    }
}


int main(int argc, char **argv) 
{
    NULL_DEV = fopen("/dev/null", "r");
    if(NULL_DEV == NULL) fprintf(stderr, "Failed to open NULL device\n");
    parseLineArgs(argc, argv);  
    //some initialization code
    init(STARTUP_FILE_NAME);
    //clear the screen
    clearScreen();
    setScreenColorsI(COLOR_WINDOW);
    //draw main menu bar
    drawMenuBar(1, 1, SCREEN_W);
    //draw main window
    drawBox(2, 1, SCREEN_H-1, SCREEN_W, documentTitle, YES, 1);
    refreshBottomView();

    if(open_file_at_startup) 
    {
        open_file_name = (char *)malloc(strlen(STARTUP_FILE_NAME)+1);
        if(!open_file_name) goto memerr;
        strcpy(open_file_name, STARTUP_FILE_NAME);
        if(!openSaveFile(OPEN, NO, open_file_name))
        {//failed to open file
            __initNewDocument();
            NEW_FILE = 1; FILE_STATE = NEW;
            checkFileExtension(open_file_name);
        }
        getScreenSize();
        refreshView();
    }
  
    if(SHOW_README) showREADMEOnStartup();
    fprintf(stdout, "\e[3;2H");
    fflush(stdout);
 
    int char_inserted = 0;
    char *ch;
    int pos, len, i;
    int refreshAll;
    while(!end) 
    {	//infinite program loop//
        ch = getKey();
        switch(ch[0]) 
        {
            case(ESC_KEY):
                if(GNU_DOS_LEVEL > 2) break;
do_esc:
                SELECTED = 0; SELECTING = 0;
                refreshView();
                break;
            case(SHIFT_DOWN):
                if(GNU_DOS_LEVEL > 3) break;
do_shift_down:
                SELECTING = 1; SELECTED = 0;
                sel_range_start.nline   = firstVisLine+selectedLine;
                sel_range_start.nchar   = selectedChar;
                sel_range_end.nline     = firstVisLine+selectedLine;
                sel_range_end.nchar     = selectedChar;
                refreshBottomView();
                break;
            case(SHIFT_UP):
                if(GNU_DOS_LEVEL > 3) break;
//do_shift_up:
                SELECTING = 0; SELECTED = 1;
                if(char_inserted) { SELECTED = 0; char_inserted = 0; }
                refreshBottomView();
                break;
            case(INS_KEY):
                INSERT        = !INSERT;
                char_inserted = 0;
                refreshBottomView();
                break;
            case(CAPS_KEY):
                CAPS          = !CAPS;
                char_inserted = 0;
                refreshBottomView();
                break;
            case(PGUP_KEY):
                if(GNU_DOS_LEVEL > 2) break;
do_pg_up:
                char_inserted = 0;
                if(firstVisLine == 0) selectedLine  = 0;
                else if(firstVisLine-totalVisLines < 0) 
                {
                     selectedLine  = firstVisLine;
                     firstVisLine  = 0;
                }
                else firstVisLine -= totalVisLines; 
     
                pos = firstVisLine+selectedLine;
                if(selectedChar > lines[pos]->charCount) selectedChar = lines[pos]->charCount;
                if(SELECTING)
                {
                    sel_range_end.nline = pos;
                    sel_range_end.nchar = selectedChar;
                }
                if(SELECTED) SELECTED = 0;
                refreshView();
                fflush(stdout);
                break;
            case(PGDOWN_KEY):
                if(GNU_DOS_LEVEL > 2) break;
do_pg_down:
                char_inserted = 0;
                firstVisLine += totalVisLines;
                if(firstVisLine+totalVisLines >= totalLines) 
                {
                    firstVisLine = totalLines-totalVisLines;
                    selectedLine = totalVisLines-1;
                }
                pos = firstVisLine+selectedLine;
                if(selectedChar > lines[pos]->charCount) selectedChar = lines[pos]->charCount;
                if(SELECTING)
                {
                    sel_range_end.nline = pos;
                    sel_range_end.nchar = selectedChar;
                }
                if(SELECTED) SELECTED = 0;
                refreshView();
                fflush(stdout);
                break;
            case(HOME_KEY):
                if(GNU_DOS_LEVEL > 2) break;
do_home:
                refreshAll        = 0;
                char_inserted     = 0;
                selectedCharCarry = 0;
                if((CTRL && GNU_DOS_LEVEL < 3) ||
                   (CTRL && GNU_DOS_LEVEL >= 3 && ch[0] == '>')) 
                {
                    selectedLine = 0;
                    firstVisLine = 0;
                    refreshAll   = 1;
                }
                else
                {
                    pos = firstVisLine+selectedLine;
                    while(pos && lines[pos-1]->linkedToNext) pos--;
                    if(pos < firstVisLine)
                    {
                        firstVisLine = pos;
                        selectedLine = 0;
                        refreshAll   = 1;
                    }
                    else
                    {
                        selectedLine = pos-firstVisLine;
                    }
                }
                selectedChar = 0;
                if(SELECTING) 
                {
                    sel_range_end.nline = firstVisLine+selectedLine;
                    sel_range_end.nchar = selectedChar;
                }
                fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
                if(SELECTED) { SELECTED = 0; }
                if(refreshAll) refreshView();
                else           refreshSelectedLine();
                fflush(stdout);
                break;
            case(END_KEY):
                if(GNU_DOS_LEVEL > 2) break;
do_end:
                refreshAll    = 0;
                char_inserted = 0;
                if((CTRL && GNU_DOS_LEVEL < 3) ||
                   (CTRL && GNU_DOS_LEVEL >= 3 && ch[0] == '<')) 
                {
                    if(totalLines <= totalVisLines) 
                    {
                        selectedLine = totalLines-1; firstVisLine = 0;
                    } 
                    else
                    {
                        firstVisLine = totalLines-totalVisLines;
                        selectedLine = totalVisLines-1;
                        refreshAll   = 1;
                        //refreshView();
                    }
                }
                else
                {
                    pos = firstVisLine+selectedLine;
                    while(lines[pos]->linkedToNext) pos++;
                    if(pos > firstVisLine+totalVisLines)
                    {
                        selectedLine = totalVisLines-1;
                        firstVisLine = pos-totalVisLines;
                        refreshAll   = 1;
                    }
                    else
                    {
                        selectedLine = pos-firstVisLine;
                    }
                }
                pos = firstVisLine+selectedLine;
                len = lines[pos]->charCount;
                //if(selectedChar == len) break;
                selectedChar = len;
                confirmEOL(pos);
                calcCharCarry(pos);
                fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
                if(SELECTING) 
                {
                    sel_range_end.nline = pos;
                    sel_range_end.nchar = selectedChar;
                }
                if(SELECTED) { SELECTED = 0; }
                if(refreshAll) refreshView();
                else           refreshSelectedLine();
                fflush(stdout);
                break;
            case(RIGHT_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_right:
                char_inserted = 0;
                pos = firstVisLine+selectedLine;
                if(WRAP_LINES)
                {
                    int oldChar = selectedChar;
                    len = lines[pos]->charCount;
                    if(selectedChar >= len ||
                        selectedChar+selectedCharCarry >= MAX_CHARS_PER_LINE-1) 
                    {
                        if((selectedLine+firstVisLine) >= (totalLines-1)) break;
                        selectedCharCarry = 0;
                        if(selectedLine == totalVisLines-1) 
                        {
                            firstVisLine++;
                            if((CTRL && GNU_DOS_LEVEL == 1) || (ALT  && GNU_DOS_LEVEL >  1))
                            {//if using CTRL, goto next word
                                gotoNextWord(pos+1, 0);
                                //means we didn't find a space till EOL
                                if(selectedChar == oldChar) selectedChar = i;
                            } else selectedChar = 0;
                            pos = firstVisLine+selectedLine;
                            //Set selection range
                            if(SELECTING) 
                            { 
                                sel_range_end.nline = pos;
                                sel_range_end.nchar = selectedChar;
                            }
                            confirmEOL(pos);
                            calcCharCarry(pos);
                            //////////////////////////////
                            refreshView();
                        }
                        else 
                        {
                            selectedLine++;
                            if((CTRL && GNU_DOS_LEVEL == 1) || (ALT  && GNU_DOS_LEVEL >  1))
                            {//if using CTRL, goto next word
                                gotoNextWord(pos+1, 0);
                                //means we didn't find a space till EOL
                                if(selectedChar == oldChar) selectedChar = i;
                            } else selectedChar = 0;
                            //fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
                            pos = firstVisLine+selectedLine;
                            if(SELECTING) 
                            { 
                                sel_range_end.nline = pos;
                                sel_range_end.nchar = selectedChar;
                            }
                            confirmEOL(pos);
                            calcCharCarry(pos);
                            //////////////////////////////
                            refreshSelectedLine();
                            refreshBottomView();
                        }
                    } 
                    else 
                    {
                        if((CTRL && GNU_DOS_LEVEL == 1) || (ALT  && GNU_DOS_LEVEL >  1))
                        {//if using CTRL, goto next word
                            gotoNextWord(pos, selectedChar+1);
                            if(selectedChar == oldChar)
                            {
                                //means we didn't find a space till EOL
                                selectedChar = i;
                                if(lines[pos]->text[selectedChar-1] == '\n') selectedChar--;
                            }                                
                        } else selectedChar++;
                        confirmEOL(pos);
                        calcCharCarry(pos);
                        ////////////////////////////////////////////////////////////
                        if(SELECTING) 
                        { 
                            sel_range_end.nline = firstVisLine+selectedLine;
                            sel_range_end.nchar = selectedChar;
                        }
                        refreshSelectedLine();
                        refreshBottomView();
                    }//if #2
                }//if #1
                if(SELECTED) { SELECTED = 0; refreshView(); }
                break;
            case(LEFT_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_left:
                char_inserted = 0;
                pos = firstVisLine+selectedLine;
                if(selectedChar == 0) 
                {
                    if(selectedLine == 0) 
                    {
                        if(firstVisLine == 0) break;
                        firstVisLine--;
                        if((CTRL && GNU_DOS_LEVEL == 1) || (ALT  && GNU_DOS_LEVEL >  1)) 
                        {//if using CTRL, goto previous word
                            gotoPrevWord(pos-1, lines[pos-1]->charCount-1);
                            if(i < 0) selectedChar = 0;
                        } 
                        else
                        {
                            selectedChar = lines[pos-1]->charCount;
                            if(lines[pos-1]->text[selectedChar-1] == '\n') selectedChar--;
                        }
                        if(SELECTING) 
                        { 
                            sel_range_end.nline = firstVisLine+selectedLine;
                            sel_range_end.nchar = selectedChar;
                        }
                        //calculate character offset
                        calcCharCarry(pos-1);
                        ////////////////////////////////////////////////////////////
                        refreshView();
                    }
                    else 
                    {
                        selectedLine--;
                        if((CTRL && GNU_DOS_LEVEL == 1) || (ALT  && GNU_DOS_LEVEL >  1))
                        {//if using CTRL, goto previous word
                            gotoPrevWord(pos-1, lines[pos-1]->charCount-1);
                            if(i < 0) selectedChar = 0;
                        }
                        else 
                        {
                            selectedChar = lines[pos-1]->charCount;
                            if(selectedChar && lines[pos-1]->text[selectedChar-1] == '\n') 
                                selectedChar--;
                        }
                        //calculate character offset
                        calcCharCarry(pos-1);
                        ////////////////////////////////////////////////////////////
                        if(selectedChar+selectedCharCarry >= MAX_CHARS_PER_LINE)
                            selectedChar--;
                        if(lines[pos-1]->charCount >= MAX_CHARS_PER_LINE)
                            fprintf(stdout, "\e[%d;%dH", selectedLine+3, 
                                    selectedChar+1+selectedCharCarry);
                        else fprintf(stdout, "\e[%d;%dH", selectedLine+3, 
                            selectedChar+2+selectedCharCarry);
                        if(SELECTING) 
                        { 
                            sel_range_end.nline = firstVisLine+selectedLine;
                            sel_range_end.nchar = selectedChar;
                            refreshView();
                        } 
                        else 
                        {
                            refreshSelectedLine();
                            refreshBottomView();
                        }
                    } //end if#2
                } 
                else 
                {
                    if((CTRL && GNU_DOS_LEVEL == 1) || (ALT  && GNU_DOS_LEVEL >  1)) 
                    {//if using CTRL, goto previous word
                        gotoPrevWord(pos, selectedChar-1);
                        if(i < 0) selectedChar = 0;
                    } 
                    else selectedChar--;
                    if(SELECTING)
                    { 
                        sel_range_end.nline = firstVisLine+selectedLine;
                        sel_range_end.nchar = selectedChar;
                    }
                    calcCharCarry(pos);
                    refreshSelectedLine();
                    refreshBottomView();
                }
                if(SELECTED) { SELECTED = 0; refreshView(); }
                break;
            case(UP_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_up:
                char_inserted = 0;
                if(selectedLine > 0) 
                {
                    selectedLine--;
                } 
                else 
                {
                    if(firstVisLine > 0) firstVisLine--;
                    else 
                    {
                        break;
                    }
                    if(!SELECTING) refreshView();
                }
                pos = firstVisLine+selectedLine;
                confirmEOL(pos);
                calcCharCarry(pos);
                //////////////////////////////////////////
                fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
                if(SELECTING)
                {
                    sel_range_end.nline = pos;
                    sel_range_end.nchar = selectedChar;
                }
                if(SELECTED) { SELECTED = 0; }
                refreshView();
                fflush(stdout);
                break;
            case(DOWN_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_down:
                char_inserted = 0;
                if((firstVisLine+selectedLine) >= totalLines-1) break;
                if(selectedLine < totalVisLines-1) 
                {
                    selectedLine++;
                } 
                else 
                { 
                    if(firstVisLine < totalLines-totalVisLines) firstVisLine++;
                    else break;
                    if(!SELECTING) refreshView();
                }
                pos = firstVisLine+selectedLine;
                confirmEOL(pos);
                calcCharCarry(pos);
                //////////////////////////////////////////
                fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
                if(SELECTING) 
                { 
                    sel_range_end.nline = pos;
                    sel_range_end.nchar = selectedChar;
                }
                if(SELECTED) { SELECTED = 0; }
                refreshView();
                fflush(stdout);
                break;
            case(BACKSPACE_KEY):
                //if(GNU_DOS_LEVEL > 3) break;
//do_backspace:
                FILE_STATE    = MODIFIED;
                char_inserted = 0;
                if(SELECTING || SELECTED) remove_selected_text(1);
                else 
                { 
                    if((CTRL && GNU_DOS_LEVEL < 4) || ALT) deletePrevWord();
                    else deletePrevChar(); 
                }
                fflush(stdout);
                break;
            case(DEL_KEY):
                if(GNU_DOS_LEVEL > 3) break;
do_del:
                FILE_STATE    = MODIFIED;
                char_inserted = 0;
                if(SELECTING || SELECTED) remove_selected_text(1);
                else 
                { 
                    if((CTRL && GNU_DOS_LEVEL < 4) || ALT) deleteNextWord();
                    else deleteNextChar();
                }
                fflush(stdout);
                break;
            case(ENTER_KEY):
                FILE_STATE    = MODIFIED;
                char_inserted = 1;
                if(SELECTING || SELECTED) remove_selected_text(1);
                insertEnter();
                fflush(stdout);
                break;
            case(TAB_KEY):
                FILE_STATE    = MODIFIED;
                if(SELECTING || SELECTED) remove_selected_text(1);
                char_inserted = 1;
                insertTab();
                fflush(stdout);
                break;
            case(SPACE_KEY):
            default:
                if(ch[0] == 'f' && ALT) 
                {
                    if(GNU_DOS_LEVEL > 1) goto do_right;//GNU key binding
                    setScreenColorsI(COLOR_WINDOW);
                    showMenu(0, open_file_name);
                } 
                else if(ch[0] == 'e' && ALT) 
                {
                    if(GNU_DOS_LEVEL > 1) break;
                    setScreenColorsI(COLOR_WINDOW);
                    showMenu(1, open_file_name);
                } 
                else if(ch[0] == 'h' && ALT) 
                {
                    if(GNU_DOS_LEVEL > 1) break;
                    setScreenColorsI(COLOR_WINDOW);
                    showMenu(3, open_file_name);
                } 
                else if(ch[0] == 'o' && ALT) 
                {
                    if(GNU_DOS_LEVEL > 1) break;
                    setScreenColorsI(COLOR_WINDOW);
                    showMenu(2, open_file_name);
                } 
                else if(ch[0] == 'b' && ALT) 
                { 
                    if(GNU_DOS_LEVEL > 1) goto do_left;//GNU key binding
                }
                else if(ch[0] == 'W' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL < 4) break;
                    FILE_STATE    = MODIFIED;
                    char_inserted = 0;
                    if(SELECTING || SELECTED) remove_selected_text(1);
                }
                else if(ch[0] == ' ' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 3) goto do_shift_down;//GNU key binding
                }
                else if(ch[0] == 'G' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 2) goto do_esc;//GNU key binding
                }
                else if(ch[0] == 'v' && ALT)
                { 
                    if(GNU_DOS_LEVEL > 2) goto do_pg_up;//GNU key binding
                }
                else if(ch[0] == 'V' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 2) goto do_pg_down;//GNU key binding
                    editMenu_Paste();
                }
                else if(ch[0] == 'C' && CTRL)
                { 
                    editMenu_Copy();
                }
                else if(ch[0] == 'X' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 4)
                    {//GNU key binding
                        setScreenColorsI(COLOR_STATUS_BAR);
                        fprintf(stdout, "\e[%d;%dH", SCREEN_H, 0);
                        fprintf(stdout, "[C-f] [C-e] [C-o] [C-h] [C-c] [C-s] [C-u] [C-g]");
                        while(1)
                        {
                            ch = getKey();
                            if(ch[0] == 'f' && CTRL)
                            { showMenu(0, open_file_name); break; }
                            else if(ch[0] == 'e' && CTRL)
                            { showMenu(1, open_file_name); break; }
                            else if(ch[0] == 'o' && CTRL)
                            { showMenu(2, open_file_name); break; }
                            else if(ch[0] == 'h' && CTRL)
                            { showMenu(3, open_file_name); break; }
                            else if(ch[0] == 'c' && CTRL)
                            { fileMenu_Exit(open_file_name); break; }
                            else if(ch[0] == 's' && CTRL)
                            { fileMenu_Save(open_file_name); break; }
                            else if(ch[0] == 'u')         { editMenu_Undo()     ; break; }
                            else if(ch[0] == 'g' && CTRL) {                       break; }
                        }//end while
                        refreshBottomView();
                        break;
                    }//end inner if
                    else editMenu_Cut();
                }
                else if(ch[0] == '/' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 4) editMenu_Undo();//GNU key binding
                }
                else if(ch[0] == '_' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 4) editMenu_Undo();//GNU key binding
                }
                else if(ch[0] == 'A' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 2) goto do_home;//GNU key binding
                    editMenu_SelectAll();
                }
                else if(ch[0] == 'Z' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 4) break;
                    editMenu_Undo();
                }
                else if(ch[0] == 'Y' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 4) //GNU key binding
                        editMenu_Paste();
                    else editMenu_Redo();
                }
                else if(ch[0] == 'R' && CTRL) 
                { 
                    editMenu_Replace();
                }
                else if(ch[0] == 'F' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 1) goto do_right;
                    editMenu_Find();
                }
                else if(ch[0] == 'E' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 2) goto do_end;
                    editMenu_ToggleSelectMode();
                }
                else if(ch[0] == 'O' && CTRL) 
                { 
                    fileMenu_Open(open_file_name); 
                }
                else if(ch[0] == 'S' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 4) editMenu_Find();
                    else fileMenu_Save(open_file_name); 
                }
                else if(ch[0] == 'N' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 1) goto do_down;
                    fileMenu_New(open_file_name);
                }
                else if(ch[0] == 'P' && CTRL)
                { 
                    if(GNU_DOS_LEVEL > 1) goto do_up;
                    fileMenu_Print(open_file_name); 
                }
                else if(ch[0] == 'Q' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 4) break;
                    fileMenu_Exit(open_file_name); 
                }
                else if(ch[0] == 'D' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 3) goto do_del;
                    deleteLine(); 
                }
                else if(ch[0] == 'K' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 3) deleteLine(); 
                }
                else if(ch[0] == 'B' && CTRL) 
                { 
                    if(GNU_DOS_LEVEL > 1) goto do_left;
                }
                else 
                {
                    FILE_STATE = MODIFIED;
                    if(SELECTING || SELECTED)
                    {
                        if(sel_range_start.nchar == sel_range_end.nchar)
                            if(sel_range_start.nline != sel_range_end.nline)
                                remove_selected_text(1);
                        SELECTED = SELECTING = 0;
                    }
                    if(CAPS)
                    {
                        if(ch[0] >= 'a' && ch[0] <= 'z') ch[0]-=32; //insert CAPITAL letter
                        else if(ch[0] >= 'A' && ch[0] <= 'Z') ch[0]+=32; //insert small letter
                        else insertChar(ch);
                    } else insertChar(ch);
                    char_inserted = 1;
                    fflush(stdout);
                    break;
                }//end if #1
                break;
        }//end switch
    }//end while
 
    restoreTerminal();
    fcloseall();
    exit(0);
 
memerr:
    fprintf(stderr, "Fatal error: Insufficient memory\n");
    EXIT();
}//end main


void deleteLine()
{
    int pos               = selectedLine+firstVisLine;
    FILE_STATE            = MODIFIED;
    int chr               = selectedChar;
    selectedChar          = 0;
    sel_range_start.nline = pos;
    sel_range_end.nline   = pos+1;
    sel_range_start.nchar = 0;
    sel_range_end.nchar   = 0;
    remove_selected_text(1);
    selectedChar          = chr;
    if(selectedChar > lines[pos]->charCount) selectedChar = lines[pos]->charCount;
    refreshView();
}

int is_whitespace(char c)
{
    if(c == ' ' || c == '\t' || c == '\v' || c == '\n' || c == '\f') return 1;
    return 0;
}

int findNextChar(int pos, int selByte)
{
    char *s = lines[pos]->text+selByte;
    char *s2 = s;
    // check for UTF-8 continuation sequence
    while((*(++s2) & 0xc0) == 0x80) ;
    return s2-s;
}

int findPrevChar(int pos, int selByte)
{
    if(selByte == 0) return 0;
    char *s = lines[pos]->text+selByte;
    char *s2 = s;
    // check for UTF-8 continuation sequence
    while((*(--s2) & 0xc0) == 0x80) ;
    return s2-s;
}

void copyInLine(int pos, int to, int from, int calcTotalChars)
{
    int len = lines[pos]->byteCount-from;
    if(len <= 0)
    {
        lines[pos]->text[to] = '\0';
    }
    else
    {
        char tmp[len+1];
        strcpy(tmp, lines[pos]->text+from);
        strcpy(lines[pos]->text+to, tmp);
    }
    if(calcTotalChars) calcTotalCharsInLine(pos);
}

int charsToBytes(int pos, int selChar)
{
    if(selChar == 0) return 0;
    int selByte = 0;
    char *s = lines[pos]->text;
    while(selChar)
    {
        // check it's not a UTF-8 continuation sequence
        if((*s & 0xc0) != 0x80) selChar--;
        selByte++;
    }
    return selByte;
}

int extendLineText(int pos, int newSize)
{
    if(lines[pos]->bytesAlloced >= newSize) return 1;
    char *s = (char *)realloc(lines[pos]->text, newSize);
    if(!s) return 0;
    lines[pos]->text = s;
    lines[pos]->text[lines[pos]->byteCount] = '\0';
    lines[pos]->bytesAlloced = newSize;
    return 1;
}

void checkLineBounds(int pos)
{
    int carry = 0;
    calcTotalCharsInLineC(pos, &carry);
    int len = lines[pos]->charCount+carry;
    if(len >= MAX_CHARS_PER_LINE)
    {
        pos++;
        move_lines_down(totalLines, pos);
        lines[pos]->linkedToNext = lines[pos-1]->linkedToNext;
        lines[pos-1]->linkedToNext = 1;
        /*
         * this is an inline modified version of charsToBytes() that takes into
         * consideration the presence of tabs within the line.
         */
        int selCharCarry = 0;
        int i = 0, j;
        char *s = lines[pos-1]->text;
        for( ; i < MAX_CHARS_PER_LINE; s++)
        {
            if(*s == '\t')
            {
                j = TABSPACES(i+selCharCarry+1);
                selCharCarry += j;
                i += j;
            }
            if((*s & 0xc0) != 0x80) i++;
        }
        j = s-lines[pos-1]->text;
        strcpy(lines[pos]->text, lines[pos-1]->text+j);
        lines[pos-1]->text[j] = '\0';
        calcTotalCharsInLine(pos-1);
        checkLineBounds(pos);
    }
    else
    {
        postDeleteWord(pos);
        calcTotalCharsInLine(pos);
    }
}

// count is the number of chars to copy. return value
// is the number of bytes actually copied.
int appendToLine(int posTo, int posFrom, int count)
{
    extendLineText(posTo, maxLen);
    char *to   = lines[posTo]->text+lines[posTo]->byteCount;
    char *from = lines[posFrom]->text;
    int  bytes = 0;
    char c;
    while(count-- > 0)
    {
        c = *from++;
        *to++ = c;
        if(!c) break;
        bytes++;
        // check for UTF-8 continuation sequence
        while((*from & 0xc0) == 0x80) *to++ = *from++, bytes++;
    }
    return bytes;
}

void postDeleteWord(int pos)
{
    int j;
    while(lines[pos]->linkedToNext) 
    {
        int carry = 0;
        calcTotalCharsInLineC(pos, &carry);
        j = MAX_CHARS_PER_LINE-(carry+lines[pos]->charCount);
        j = appendToLine(pos, pos+1, j);
        copyInLine(pos+1, 0, j, 1);
        pos++;
        if(pos >= totalLines) break;
    }
    if(lines[pos]->charCount == 0) 
    {
        lines[pos-1]->linkedToNext = 0;
        move_lines_up(pos, totalLines-1);
    }
}

void deleteNextWord() 
{
    int i, j, pos = firstVisLine+selectedLine;
    int       chr = charsToBytes(pos, selectedChar);
    int       oF  = firstVisLine;
    int       oS  = selectedLine;
    int       oC  = selectedChar;
    if(selectedChar >= lines[pos]->charCount) 
    {
        if(pos >= totalLines-1) return;
        lines[pos]->linkedToNext = 1; pos++; chr = 0;
        selectedChar = 0; firstVisLine++;
    }
    FILE_STATE = MODIFIED;
    i = findNextChar(pos, chr);  
    while(is_whitespace(lines[pos]->text[i])) i++;

    for( ; i < lines[pos]->byteCount; i++)
    {
        if(is_whitespace(lines[pos]->text[i])) break;
        else
        {
            int j = undoAddUtfChar(UNDO_ACTION_DELETE, pos, i, NULL);
            i += (j-1);
            selectedChar++;
        }
    }
    
    if(i >= lines[pos]->byteCount) lines[pos]->text[chr] = '\0';
    else copyInLine(pos, chr, i, 1);
  
    firstVisLine = oF; selectedLine = oS; selectedChar = oC;
    //calcTotalCharsInLine(pos);
    postDeleteWord(firstVisLine+selectedLine);
    //checkAllTabs();
    refreshView();
}

void deletePrevWord() 
{
    int i, j, pos = firstVisLine+selectedLine;
    int       chr = charsToBytes(pos, selectedChar);
    int       oF  = firstVisLine;
    int       oS  = selectedLine;
    int       oC  = selectedChar;
    int deleteNL  = 0;
    if(selectedChar <= 0) 
    {
        if(pos <= 0) return;
        pos--;
        lines[pos]->linkedToNext = 1;
        chr = lines[pos]->charCount;
        selectedChar = chr;
        firstVisLine--;
        deleteNL = 1;
    }
    FILE_STATE = MODIFIED;
    i = chr;
    while(i && is_whitespace(lines[pos]->text[i])) i--;
    while(i && !is_whitespace(lines[pos]->text[i]))
    {
        i--;
        if((lines[pos]->text[i] & 0xc0) != 0x80) selectedChar--;
    }
    if(i < 0) i = 0;

    for( ; i < chr; i++)
    {
        int j = undoAddUtfChar(UNDO_ACTION_DELETE, pos, i, NULL);
        i += (j-1);
    }
    
    if(i < 0) i = 0;
    else copyInLine(pos, i, chr, 1);
  
    firstVisLine = oF; selectedLine = oS; selectedChar = oC;
    //calcTotalCharsInLine(pos);
    postDeleteWord(firstVisLine+selectedLine);
    calcCharCarry(pos);
    refreshView();
}

void deleteNextChar() 
{
    int i, pos = firstVisLine+selectedLine;
    char deleteNL = 0;
    char refreshAll = 0;
    char c = lines[pos]->text[selectedChar];
    if(c == '\n' || c == '\0') deleteNL = 1;
    undoAddUtfChar(UNDO_ACTION_DELETE, pos, selectedChar, NULL);

    int newChar = findNextChar(pos, selectedChar);    
    copyInLine(pos, selectedChar, newChar, 1);
    if(deleteNL)
    {
        if(pos == totalLines-1) return;
        //is the next line an empty line?
        if(lines[pos+1]->charCount == 0 || lines[pos+1]->text[0] == '\n')
        {
            lines[pos]->linkedToNext = 0;
            move_lines_up(pos+1, totalLines-1);
            refreshAll = 1;
        }
        else
        {
            lines[pos]->linkedToNext = 1;
        }
    }

    if(lines[pos]->linkedToNext) 
    {
        postDeleteWord(pos);
        refreshAll = 1;
    }

    if(refreshAll) refreshView();
    else refreshSelectedLine();
}


void deletePrevChar()
{
    int i, pos;
    int selChar;
  
    if(selectedChar == 0) 
    {
        if(selectedLine == 0)
        {
            if(firstVisLine == 0) return;
            firstVisLine--;      
        }
        else
        {
            selectedLine--;
        }
        pos = firstVisLine+selectedLine;
        lines[pos]->linkedToNext = 1;
        selChar = lines[pos]->byteCount;
        selChar = findPrevChar(pos, selChar);
        selectedChar = lines[pos]->charCount-1;
    }
    else
    {
        pos = firstVisLine+selectedLine;
        selectedChar--;
        selChar = charsToBytes(pos, selectedChar);
    }
    undoAddUtfChar(UNDO_ACTION_DELETE, pos, selChar, NULL);
    char c = lines[pos]->text[selChar];
    i = findNextChar(pos, selChar);
    if(lines[pos]->text[i] == '\0')
    {
        lines[pos]->text[0] = '\0';
        lines[pos]->charCount = 0;
        lines[pos]->byteCount = 0;
    }
    else copyInLine(pos, selChar, i, 1);
    
    //calcTotalCharsInLine(pos);
    postDeleteWord(pos);
    if(c == '\t') calcCharCarry(pos);

    if(lines[pos]->linkedToNext) refreshView();
    else refreshSelectedLine();
}

void insertEnter() 
{
    int i = 0;
    int pos = firstVisLine+selectedLine;
    int autoIndentLen = 0;

    if(AUTO_INDENT)
    {
        for( ; i < lines[pos]->byteCount; i++)
        {
            char c = lines[pos]->text[i];
            if(is_whitespace(c))
            {
                autoIndentStr[i] = lines[pos]->text[i];
            }
            else break;
        }
        autoIndentStr[i] = '\0';
        autoIndentLen = i;
    }
    
    int j = charsToBytes(pos, selectedChar);
    undoAddChar(UNDO_ACTION_INSERT, pos, j, '\n');
    int count = lines[pos]->byteCount-j+i;
    char *newLine = malloc(count+1);
    if(!newLine) return;
    newLine[0] = '\0';
    if(AUTO_INDENT)
    {
        strcpy(newLine, autoIndentStr);
        int k = j;
        while(i--)
        {
            undoAddChar(UNDO_ACTION_INSERT, pos+1, k, autoIndentStr[k-j]);
            k++;
        }
    }
    if(lines[pos]->text[j]) strcat(newLine, lines[pos]->text+j);
    lines[pos]->text[j  ] = '\n';
    lines[pos]->text[j+1] = '\0';
    calcTotalCharsInLine(pos);
    pos++;
    move_lines_downl(totalLines+1, pos, newLine);
    lines[pos]->linkedToNext = lines[pos-1]->linkedToNext;
    lines[pos-1]->linkedToNext = 0;
    if(count >= maxLen)
    {
        pos++;
        move_lines_down(totalLines+1, pos);
        lines[pos]->linkedToNext = lines[pos-1]->linkedToNext;
        lines[pos-1]->linkedToNext = 1;
        j = charsToBytes(pos-1, MAX_CHARS_PER_LINE);
        strcpy(lines[pos]->text, lines[pos-1]->text+j);
        lines[pos-1]->text[j] = '\0';
        calcTotalCharsInLine(pos-1);
    }
    //calcTotalCharsInLine(pos);
    postDeleteWord(pos);
    if(selectedLine == totalVisLines-1) firstVisLine++;
    else selectedLine++;
    selectedChar = autoIndentLen;
    calcCharCarry(pos);
    refreshView();
}


void insertTab() 
{
    static char *t = "\t";
    insertChar(t);
    refreshBottomView();
    return;
}

int getUtfCharLength(char *ch)
{
    int len = 0;
    if((*ch & 0xc0) != 0x80) ch++, len++;
    while((*ch & 0xc0) == 0x80) ch++, len++;
    return len;
}

/*
 * replace one multibyte UTF-8 char with another, checking to see if the two chars
 * are of the same length. If not, do the necessary shuffling of bytes before replacing.
 */
void replaceChar(int pos, int i, char *ch, int calcTotalChars)
{
    char *orig = lines[pos]->text+i;
    int len1 = getUtfCharLength(orig);
    int len2 = getUtfCharLength(ch);
    int diff = len2-len1;
    if(diff == 0)
    {
        while(len1--) *orig++ = *ch++;
    }
    else if(diff < 0)
    {
        while(len2--) *orig++ = *ch++;
        int to = orig-lines[pos]->text;
        int from = i+len1;
        copyInLine(pos, to, from, calcTotalChars);
    }
    else
    {
        if(!extendLineText(pos, lines[pos]->byteCount+diff+1)) return;
        copyInLine(pos, i+len2, i+len1, 0);
        while(len2--) *orig++ = *ch++;
        if(calcTotalChars) calcTotalCharsInLine(pos);
    }
}

void insertChar(char *ch) 
{
    int pos = firstVisLine+selectedLine;
    int i   = charsToBytes(pos, selectedChar);
    int refreshAll = 0;

    if(WRAP_LINES)
    {
        if(INSERT) 
        {//replace current character
            char c1 = lines[pos]->text[i];
            char c2 = *ch;
            undoAddUtfChar(UNDO_ACTION_REPLACE, pos, i, ch);
            replaceChar(pos, i, ch, 1);
            if(c1 == '\t' || c2 == '\t')
            {
                //calcTotalCharsInLine(pos);
                checkLineBounds(pos);
                refreshAll = 1;
            }
        }
        else
        {
            undoAddUtfChar(UNDO_ACTION_INSERT, pos, i, ch);
            int len2 = getUtfCharLength(ch);
            if(!extendLineText(pos, lines[pos]->byteCount+len2+1)) return;
            char *orig = lines[pos]->text+i;
            copyInLine(pos, i+len2, i, 0);
            while(len2--) *orig++ = *ch++;
            //calcTotalCharsInLine(pos);
            checkLineBounds(pos);
            if(lines[pos]->linkedToNext) refreshAll = 1;
        }
    }
    selectedChar++;
    if(selectedChar+selectedCharCarry >= MAX_CHARS_PER_LINE)
    {
        if(selectedLine == totalVisLines-1) firstVisLine--;
        else selectedLine++;
        selectedChar = 0;
        refreshAll = 1;
    }
    calcCharCarry(firstVisLine+selectedLine);
    if(refreshAll) refreshView();
    else refreshSelectedLine();
}

void refreshSelectedLine() 
{
    int pos = firstVisLine+selectedLine;
    refreshViewLines(pos, pos, selectedLine);
    refreshBottomView();
    fflush(stdout);
}

void outputEmptyLine(int i)
{
    fprintf(stdout, "\e[%d;%dH", i+3, 2);
    fprintf(stdout, "%*s", MAX_CHARS_PER_LINE, " ");
}

void padLineWithSpaces(int len)
{
    if(len < MAX_CHARS_PER_LINE)
        fprintf(stdout, "%*s", MAX_CHARS_PER_LINE-len, " ");
}

void putuchar(int pos, int index, int *carry)
{
    // check for tabs
    if(lines[pos]->text[index] == '\t')
    {
        int k = TABSPACES(index+(*carry)+1);
        (*carry) += k;
        if(k) fprintf(stdout, "%*s", k+1, " ");
        return;
    }

    static char c[5];
    memset(c, 0, 5);
    c[0] = lines[pos]->text[index];
    /*
     * TODO: Is it really good to inhibit newline output
     *       globally like this? All functions calling this
     *       function will result in newline chars not being
     *       output. Is it healthy?
     */
    if(c[0] == '\n') return;
    if ((c[0] & mask[0]) == mask[0]) c[1] = lines[pos]->text[index+1];
    if ((c[0] & mask[1]) == mask[1]) c[2] = lines[pos]->text[index+2];
    if ((c[0] & mask[2]) == mask[2]) c[3] = lines[pos]->text[index+3];
    c[4] = '0';
    fprintf(stdout, "%s", c);
}

int strStartsWith(const char *pre, const char *str)
{
    return strncmp(pre, str, strlen(pre)) == 0;
}

int strStartsWithL(const char *pre, const char *str, int prelen)
{
    return strncmp(pre, str, prelen) == 0;
}

char *specialCharsString = "[]{}()<>:;,*+-=%_!#$^&`~\\";

int isBraceChar(char c)
{
    char *s = specialCharsString;
    while(*s)
    {
        if(c == *s++) return 1;
    }
    return 0;
}

int isQuoteChar(char c)
{
    if(c == '"' || c == '\'') return 1;
    return 0;
}

int isSpaceChar(char c)
{
    if(c == ' ' || c == '\t' || c == '\f' || c == '\r' || c == '\v') return 1;
    return 0;
}

int indexOf(char *str, char *substr)
{
    if(!str || ! substr) return -1;
    char *s = strstr(str, substr);
    return s ? s-str : -1;
}

/********************************************************
 * This function colorizes the text according to the
 * predefined highlight colors. It searches for keywords,
 * braces, comments and strings in each line.
 * ******************************************************/
void refreshSelectedLineInColor(int pos, int *incomment)
{
    int j, carry = 0;
    char STRING_STARTED = 0;//bool to indicate if we are inside a string
    char quoteChar;
    //char inKeyword = 0;
    //int printed_chars = 0;
    
    char *line = lines[pos]->text;
    int len = lines[pos]->charCount;
    int mcstart = indexOf(line, curmodule->mlCommentStart);
    int mcend   = indexOf(line, curmodule->mlCommentEnd  );
    int scstart = indexOf(line, curmodule->slCommentStart);
    setScreenColorsI(COLOR_WINDOW);
    
    for(j = 0; j < lines[pos]->byteCount; j++) 
    {
       // check it is not a UTF-8 continuation sequence
       if((lines[pos]->text[j] & 0xc0) == 0x80) continue;
       // is it the start of a multiline comment?
       if(mcstart >= 0 && j == mcstart)
       {
            setScreenColors(COLOR_HCOMMENT, BG_COLOR[COLOR_WINDOW]);
            putuchar(pos, j, &carry);
            *incomment = 1;
            continue;
       }
       // is it the end of a multiline comment?
       else if(mcend >= 0 && j == mcend)
       {
            char *mc = curmodule->mlCommentEnd;
            while(*mc++) putuchar(pos, j++, &carry);
            j--;
            setScreenColorsI(COLOR_WINDOW);
            *incomment = 0;
            continue;
       }
       // are we in a comment?
       else if(*incomment)
       {
           putuchar(pos, j, &carry);
           continue;
       }
       // is it the start of a single-line comment?
       else if(scstart >= 0 && j == scstart)
       {
            setScreenColors(COLOR_HCOMMENT, BG_COLOR[COLOR_WINDOW]);
            for( ; j < lines[pos]->byteCount; j++) 
            {
                // check it is not a UTF-8 continuation sequence
                if((lines[pos]->text[j] & 0xc0) != 0x80) putuchar(pos, j, &carry);
            }
            break;
       }
       // are we inside a string?
       if(STRING_STARTED)
       {
           putuchar(pos, j, &carry);
           if(quoteChar == line[j])
           {
               STRING_STARTED = 0;
               setScreenColorsI(COLOR_WINDOW);
           }
       }
       else
       {
           char c = line[j];
           if(c == '\n') continue;
           if(isBraceChar(c) || isSpaceChar(c))
           {
               setScreenColors(COLOR_HBRACES, BG_COLOR[COLOR_WINDOW]);
               putuchar(pos, j, &carry);
               //setScreenColors(COLOR_HPARAMETERS, BG_COLOR[COLOR_WINDOW]);
               setScreenColorsI(COLOR_WINDOW);
           }
           else if(isQuoteChar(c))
           {
                if(STRING_STARTED)
                {
                    STRING_STARTED = 0;
                    putuchar(pos, j, &carry);
                    //setScreenColors(COLOR_HPARAMETERS, BG_COLOR[COLOR_WINDOW]);
                    setScreenColorsI(COLOR_WINDOW);
                }
                else
                {
                    STRING_STARTED = 1;
                    quoteChar = c;
                    setScreenColors(COLOR_HSTRING, BG_COLOR[COLOR_WINDOW]);
                    putuchar(pos, j, &carry);
                }
           }
           else
           {
                int i;
                if(isKeyword(pos, j, &i))
                {
                    setScreenColors(COLOR_HKEYWORD, BG_COLOR[COLOR_WINDOW]);
                    while(i--) putuchar(pos, j++, &carry);
                    j--;
                    //setScreenColors(COLOR_HPARAMETERS, BG_COLOR[COLOR_WINDOW]);
                    setScreenColorsI(COLOR_WINDOW);
                }
                else
                {
                    while(i--) putuchar(pos, j++, &carry);
                    j--;
                }
           }
       }        
    }
    padLineWithSpaces(len+carry);
    fflush(stdout);
}

//This function tells refreshSelectedLineInColor() whether
//we are standing on a keyword (to give it keyword color)
//or not. If yes, it returns the length of the keyword.
int isKeyword(int pos, int start, int *wordlen)
{
    int i = 0;
    int result = 0;
    char *p = lines[pos]->text+start;
    while(*p && !isSpaceChar(*p) && !isBraceChar(*p)) p++, i++;
    if(p[-1] == '\n') i--;
    char word[i+1];
    memcpy(word, lines[pos]->text+start, i);
    word[i] = '\0';
    *wordlen = strlen(word);

    for(i = 0; i < curmodule->keywordCount; i++)
    {
        if(*wordlen != strlen(curmodule->keywords[i])) continue;
        if(curmodule->caseSensitive) result = strcmp(word, curmodule->keywords[i]);
        else result = strcasecmp(word, curmodule->keywords[i]);
        if(result == 0) return 1;
    }
    return 0;
}

void refreshView() 
{
    //turn the cursor off
    fprintf(stdout, "\e[?25l");
    int i;
    if(totalLines-firstVisLine < totalVisLines && totalLines > totalVisLines)
    {
        i = firstVisLine;
        firstVisLine = totalLines-totalVisLines;
        selectedLine += i-firstVisLine;
    }
    setScreenColorsI(COLOR_WINDOW);
    if(WRAP_LINES)
    {
        if(totalLines < totalVisLines) 
        {
            refreshViewLines(0, totalLines-1, 0);
            setScreenColorsI(COLOR_WINDOW);
            for(i = totalLines; i < totalVisLines; i++) outputEmptyLine(i);
        }
        else 
        {
            refreshViewLines(firstVisLine, firstVisLine+totalVisLines-1, 0);
        }
        fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
    }
    drawMenuBar(1, 1, SCREEN_W);
    drawBox(2, 1, SCREEN_H-1, SCREEN_W, documentTitle, NO, 1);
    drawScrollBar();
    refreshBottomView();
    //turn the cursor on
    fprintf(stdout, "\e[?25h");
    fflush(stdout);
}//end refreshView()

        
int commentStatus(int pos)
{
    if(!curmodule->mlCommentStart) return 0;
    if(pos) pos--;
    for( ; pos >= 0; pos--)
    {
        char *ce = strstr(lines[pos]->text, curmodule->mlCommentEnd);
        char *cs = strstr(lines[pos]->text, curmodule->mlCommentStart);
        if(ce && cs)
        {
            if(cs >= ce) return 1;
            return 0;
        }
        if(cs) return 1;
    }
    return 0;
}


void refreshViewLines(int start, int end, int startOutputAt)
{
    int swap = 0;
    int i;
    if(SELECTING || SELECTED) 
    {
        //////////////////////////////////////////
        //If in Selecting Mode
        //////////////////////////////////////////
        if(sel_range_start.nline > sel_range_end.nline) { swap = 1; swap_lines(); }
        else if(sel_range_start.nline == sel_range_end.nline &&
        sel_range_start.nchar > sel_range_end.nchar) { swap = 2; swap_chars(); }
        for(i = startOutputAt; start <= end; i++, start++)
        {
            fprintf(stdout, "\e[%d;%dH", i+3, 2);
            int len = lines[start]->charCount;
            int j, k, l;
            if(start == sel_range_start.nline)
            { 
                k = sel_range_start.nchar; l = len-1;
            } 
            else if(start == sel_range_end.nline) 
            { 
                k = 0; l = sel_range_end.nchar;
            } 
            else 
            { 
                k = 0; 
                if(start > sel_range_start.nline && start < sel_range_end.nline)
                     l = len-1;
                else l = -1;
            }
            int carry = 0;
            for(j = 0; j < lines[start]->byteCount; j++) 
            {
                if(j >= k && j <= l) setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                else setScreenColorsI(COLOR_WINDOW);
                // check it is not a UTF-8 continuation sequence
                if((lines[start]->text[j] & 0xc0) != 0x80) putuchar(start, j, &carry);
            }
            len += carry;
            setScreenColorsI(COLOR_WINDOW);
            if(len < MAX_CHARS_PER_LINE)
                fprintf(stdout, "%*s", MAX_CHARS_PER_LINE-len, " ");
        }
        if(swap == 1) swap_lines();//return them back to normal
        if(swap == 2) swap_chars();//return them back to normal
    } 
    else 
    {
        //////////////////////////////////////////
        //If in Regular Mode
        //////////////////////////////////////////
        int incomment = commentStatus(start);
        int j;
        setScreenColorsI(COLOR_WINDOW);
        for(i = startOutputAt; start <= end; i++, start++)
        {
            fprintf(stdout, "\e[%d;%dH", i+3, 2);
            int carry = 0;
            int len = lines[start]->charCount;
            if(AUTO_HIGHLIGHTING)
                refreshSelectedLineInColor(firstVisLine+i, &incomment);
            else
            {
                for(j = 0; j < lines[start]->byteCount; j++) 
                {
                    // check it is not a UTF-8 continuation sequence
                    if((lines[start]->text[j] & 0xc0) != 0x80) putuchar(start, j, &carry);
                }
                padLineWithSpaces(len+carry);
            }
        }
    }
}



void drawScrollBar() 
{
    int h = SCREEN_H-5;
    int i;
    setScreenColorsI(COLOR_MENU_BAR);
    for(i = 0; i <= h; i++)
        fprintf(stdout, "\e[%d;%dH ", i+3, SCREEN_W);
    double h2;
    h2  = firstVisLine+selectedLine+1;
    h2 /= totalLines;
    h2 *= h;
    if(h2 < 0) h2 = 0;
    if(h2 > (SCREEN_H-5)) h2 = SCREEN_H-5;
    setScreenColorsI(COLOR_WINDOW);
    fprintf(stdout, "\e[%d;%dH%c", (int)(h2)+3, SCREEN_W, 177);
    fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
}

void refreshBottomView() 
{
    setScreenColorsI(COLOR_STATUS_BAR);
    fprintf(stdout, "\e[%d;%dH", SCREEN_H, 0);
    printf("%*s", SCREEN_W, " ");
    fprintf(stdout, "\e[%d;%dH", SCREEN_H, SCREEN_W-19);
    printf("| LINE:%-3d COL:%-3d", firstVisLine+selectedLine+1, selectedChar+1);  
    if(CAPS     ) fprintf(stdout, "\e[%d;%dHCAPS", SCREEN_H, SCREEN_W-24);
    if(INSERT   ) fprintf(stdout, "\e[%d;%dHINS" , SCREEN_H, SCREEN_W-28);
    if(SELECTING) fprintf(stdout, "\e[%d;%dHSEL" , SCREEN_H, SCREEN_W-32);
    fprintf(stdout, "\e[%d;%dH", SCREEN_H, 2);
    switch(FILE_STATE) 
    {
        case(MODIFIED): printf("Modified"); break;
        case(NEW):      printf("New");      break;
        case(SAVED):    printf("Saved");    break;
        case(OPENED):   printf("Opened");   break;
        case(IDLE):     printf("Idle");     break;
    }
    if(selectedChar+selectedCharCarry > MAX_CHARS_PER_LINE-1)
         fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+1+selectedCharCarry);
    else fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
    fflush(stdout);
}

/***************************************
 * drawMenuBar(): 
 * Procedure to draw the main menu bar.
 ***************************************/
void drawMenuBar(int x, int y, int w) 
{
    setScreenColorsI(COLOR_MENU_BAR);
    fprintf(stdout, "\x1b[%d;%dH", x, y);		//reposition the cursor
    int i,j, lastChar=y;
    for(i = 0; i < w; i++) fputc(' ', stdout);	//Draw empty menu bar
    fprintf(stdout, "\x1b[%d;%dH", x, y);		//reposition the cursor

    for(i = 0; i < totalMainMenus; i++) 
    {
        j=0; lastChar++;
        fprintf(stdout, " ");
        while(menu[i][j] != '\0') 
        {
            if(menu[i][j] == '&') 
            {	//turn on underline feature to print the shortcut key
                //fprintf(stdout, "\x1b[4m%c", menu[i][j+1]);
                //fprintf(stdout, "\x1b[24m"); //then turn it off
                setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                fprintf(stdout, "%c", menu[i][j+1]);
                setScreenColorsI(COLOR_MENU_BAR);
            }
            else
                fprintf(stdout, "%c", menu[i][j+1]);	//print normal chars (other than the
            lastChar++; j++;					//shortcut key)
        }
        fprintf(stdout, " ");
    }
    setScreenColorsI(COLOR_WINDOW);
    fprintf(stdout, "\x1b[24m");
    fflush(stdout);
}

void catchSignals() 
{
    if(signal(SIGINT, sighandler) == SIG_ERR) 
    {
      printf("Error interrupting SIGINT.\n");
      exit(1);
    }
    if(signal(SIGQUIT, sighandler) == SIG_ERR) 
    {
      printf("Error interrupting SIGQUIT.\n");
      exit(1);
    }
    if(signal(SIGABRT, sighandler) == SIG_ERR) 
    {
      printf("Error interrupting SIGABRT.\n");
      exit(1);
    }
    if(signal(SIGTERM, sighandler) == SIG_ERR) 
    {
      printf("Error interrupting SIGTERM.\n");
      exit(1);
    }
    if(signal(SIGTSTP, sighandler) == SIG_ERR) 
    {
      //exit(1);
    }
}
      
/***************************************
 * drawBox(): 
 * Procedure to draw a box with the given
 * coordinates, title, and a flag
 * indicating whether to clear the window
 * area or not (passed as YES or NO).
 ***************************************/
void drawBox(int x1, int y1, int x2, int y2, char *title, int clearArea, int isMainWindow)
{
  char spaces[y2-y1];
  int i;
  for(i = 0; i < y2-y1-1; i++) spaces[i] = ' ';
  spaces[i] = '\0';
  //Draw the box first//
  setScreenColorsI(COLOR_WINDOW);
  fprintf(stdout, "\x1b[%d;%dH", x1, y1);	//control sequence to move cursor
  /* NOTE: this is a TERRIBLE hack! but it does the following:
   *       "\e)0" will define G1 charset to be "VT100 Graphics Mapping"
   *       "\x0e" a.k.a. ^N, activates G1 charset.
   */
  fprintf(stdout, "\e)0\x0e");
  fflush(stdout);

  putchar(ULC);					//print the upper-left corner
  for(i = 0; i < (y2-y1)-1; i++) 
  {
    putchar(HB);				//print the horizontal upper bar
  }
  putchar(URC);  				//print the upper-right corner
  putchar('\n');			//finished window top, make a new line
  
  for(i = 0; i < (x2-x1)-1; i++) 
  {
    fprintf(stdout, "\x1b[%d;%dH", x1+i+1, y1);	//move cursor to left window edge
    if(clearArea == YES) 
    {
      fprintf(stdout, "%c%s%c", VB, spaces, VB);
    } 
    else 
    {//print left VB, no spaces and right vertical bar
      fprintf(stdout, "%c\x1b[%d;%dH%c", VB, x1+i+1, y2, VB);
    }
  }

  fprintf(stdout, "\x1b[%d;%dH", x2, y1);	//control sequence to move cursor
  putchar(LLC);				//print the lower-left corner
  for(i = 0; i < (y2-y1)-1; i++) 
  {
    putchar(HB);			//print the horizontal lower bar
  }
  putchar(LRC);  
  /* NOTE: this is a TERRIBLE hack! but it does the following:
   *       "\x0f" a.k.a. ^O, activates G0 charset.
   */
  fprintf(stdout, "\x0f");
  //fflush(stdout);  
  
  //Then put on the box title, if any//
  if(title != NULL) 
  {
    int tmp1=(y2-y1)/2;
    int tmp2=strlen(title)/2;
    fprintf(stdout, "\x1b[%d;%dH%s",	//move the cursor
		    x1,			//to the top
		    y1+tmp1-tmp2,	//and center of the box
		    title);		//to print this title
    if(FILE_STATE == MODIFIED && isMainWindow) fputc('*', stdout);
  }
  fflush(stdout);
}


void move_lines_up(int first, int last)
{
    if(first == last) return;
    int i;
    for(i = first; i < last; i++)
    {
        copyLineStruct(i, i+1);
    }
    if(lines[last]->text) free(lines[last]->text);
    free(lines[last]);
    lines[last] = NULL;
    totalLines--;
}

// shift lines up by the difference between first and last lines
void move_lines_upd(int first, int diff)
{
    int i = first;
    if(diff == 0) return;
    if(first == totalLines-diff) goto finish;
    for( ; i < totalLines-diff; i++)
    {
        copyLineStruct(i, i+diff);
    }

finish:
    for( ; i < totalLines; i++)
    {
        free(lines[i]);
        lines[i] = NULL;
    }
    totalLines -= diff;
}

void move_lines_downl(int first, int last, char *newLineText)
{
    if(first == last) return;
    int i;
    for(i = first; i > last; i--)
    {
        copyLineStruct(i, i-1);
    }
    totalLines++;
    if(newLineText)
    {
        lines[last] = allocLineStruct();
        lines[last]->text = newLineText;
        calcTotalCharsInLine(last);
    }
    else lines[last] = allocLineStructB(maxLen);
}

void move_lines_down(int first, int last)
{
    move_lines_downl(first, last, NULL);
}

void calcCharCarry(int pos)
{
    selectedCharCarry = 0;
    int i = 0, j;
    char *s = lines[pos]->text;
    for( ; i < selectedChar; s++)
    {
        if(*s == '\t')
        {
            j = TABSPACES(i+selectedCharCarry+1);
            selectedCharCarry += j;
        }
        if((*s & 0xc0) != 0x80) i++;
    }
}

